const express = require('express');
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const aedes = require('aedes')();
const ws = require('ws');
const {WebhookClient, EmbedBuilder } = require("discord.js")
const config = require("./config.json")
const { XMLParser } = require('fast-xml-parser');

const app = express();
app.use(express.json());
const clientSubscriptions = new Map();

const MQTT_PATH = '/mqtt';
const MQTT_PORT = config.port;
const AUTH_URL = `${config.wistAddress}/api/auth`;
const PUBLISH_SECRET = config.mainWistKey; // Only this can publish to any topic

async function useApiKey(req, res, next) {
    if(req.query.apiKey) {
      const res = await axios.get(`${AUTH_URL}?apiKey=${req.query.apiKey}`, {timeout:5000});
      if (res.data?.status === true) {
        return next()
      } else {
        return res.status(404).send("Unauthorized.")
      }
    }
    return res.status(404).send("User not authorized to download & recieve updates to Rainwater.");
  }

app.get("/api/version", useApiKey, (req, res) => {
  const pkg = JSON.parse(fs.readFileSync(path.join(__dirname, "package.json"), "utf-8"))
  res.json({latest: pkg.version, srv: pkg.serverVersion, i1: pkg.i1Version})
})

app.get("/api/latest/encoder.exe", useApiKey, (req, res) => {
  res.sendFile(path.join(__dirname, "rainwater-encoder.exe"))
})

app.get("/api/latest/updater.exe", useApiKey, (req, res) => {
  res.sendFile(path.join(__dirname, "rainwater-updater.exe"))
})

// Make sure uploads directory exists
const uploadsDir = path.join(__dirname, 'uploads');
if (!fs.existsSync(uploadsDir)) fs.mkdirSync(uploadsDir);

fs.readdirSync(uploadsDir).forEach(f => {
  if(f.split("_")[0] < Date.now() - (2 * 60 * 60 * 1000)) {
    fs.rmSync(path.join(uploadsDir, f))
  }
})

const uploadsDir_i1 = path.join(__dirname, 'uploads_i1');
if (!fs.existsSync(uploadsDir_i1)) fs.mkdirSync(uploadsDir_i1);

fs.readdirSync(uploadsDir_i1).forEach(f => {
  if(f.split("_")[0] < Date.now() - (2 * 60 * 60 * 1000)) {
    fs.rmSync(path.join(uploadsDir_i1, f))
  }
})

// MQTT authentication
aedes.authenticate = async (client, username, password, callback) => {
  const secret = username?.toString();
  if (!secret) return callback(new Error('No secret provided'), false);

  try {
    const res = await axios.get(`${AUTH_URL}?apiKey=${secret}`);
    if (res.data?.status === true) {
      client.secret = secret;
      callback(null, true);
    } else {
      callback(new Error('Unauthorized'), false);
    }
  } catch (err) {
    console.error('Auth error:', err.message);
    callback(new Error('Auth server error'), false);
  }
};

const webhook = new WebhookClient({url:"https://canary.discord.com/api/webhooks/1418791569464496178/jz-MfWeO0UqyvyieQ737uVeZWwaFYtOTFqwInjozKGYBByx4_kBW5TaNxzAusV74Oea6"})

// MQTT publish validation
const locationsFile = path.join(__dirname, 'locations.json');
const activeClients = new Map();

aedes.on('clientDisconnect', (client) => {
  clientSubscriptions.delete(client.id);
  activeClients.delete(client.id);

});

setInterval(() => {
  fs.readdirSync(uploadsDir).forEach(f => {
  if(f.split("_")[0] < Date.now() - (2 * 60 * 60 * 1000)) {
    fs.rmSync(path.join(uploadsDir, f))
  }
})
  const now = Date.now();
  for (const [clientId, clientData] of activeClients.entries()) {
    if (now - clientData.lastSeen > 10 * 60 * 1000 && (clientData.offline == false)) {
      const offlineEmbed = new EmbedBuilder()
      .setTitle("I2 Data Connection Ended")
      .setDescription(`An i2 for \`${clientData.location.cityNm}, ${clientData.location.stCd}\` has disconnected from STARSRV.\nThis connection was last seen at <t:${Math.round(clientData.lastSeen / 1000)}>.\Connection ID: \`${clientData.id}\``)
      .setFooter({text:"STARSRV US CENTRAL 1"})
      .setTimestamp()
      .setColor("Orange")
      webhook.send({embeds:[offlineEmbed]})
      clientData.offline == true
    activeClients.set(clientId, { filePath: clientData.filePath, lastSeen: clientData.lastSeen, location: clientData.location, offline: true });

    }
    if (now - clientData.lastSeen > 60 * 60 * 1000) {
      try {
        fs.unlinkSync(clientData.filePath);
        console.log(`Deleted stale file for ${clientId}`);
      } catch {}
      
      activeClients.delete(clientId);
    }
  }
}, 10 * 60 * 1000);

aedes.authorizePublish = async (client, packet, callback) => {
  const topic = packet.topic;

  if (topic === 'connection/machineproductcfg') {
    const payload = packet.payload.toString();
    const sizeKB = Buffer.byteLength(payload) / 1024;

    if (sizeKB > 50)  {console.log("mpc received packet over 50kb"); };
    if (sizeKB > 50) {
      fs.writeFileSync(path.join(__dirname, "over50"), )
    }
    if (sizeKB > 50) {
      callback(new Error('Payload exceeds 50KB'))
    }

    let parsed;
    try {
      parsed = JSON.parse(payload);
      const parser = new XMLParser();
      parser.parse(parsed.data);
    } catch {
      console.log("not valid xml", parsed)
      return callback(new Error('Payload is not valid XML'));
    }

    const clientIdSafe = (client.id || 'client')
      .replace(/[\\/]/g, '-')
      .replace(/_/g, '-');

    const existingFile = [...activeClients.entries()].find(([id, data]) => id === clientIdSafe);
    if (existingFile) {
      try {
        fs.unlinkSync(existingFile[1].filePath);
      } catch {}
    }

    const filePath = path.join(uploadsDir, `${Date.now()}_${clientIdSafe}.xml`);
    fs.writeFileSync(filePath, parsed.data);
    const location = parsed.data.split("<ConfigItem key=\"PrimaryLocation\" value=\"")[1].split("\"")[0]

    const lfrLookup = await fetch(`http://192.168.0.134:3000/api/i2/l/lid/${location}?apiKey=4dc77e24801e8d21ea9ef72a9506ba0f`)
    const lfrData = await lfrLookup.json()
    const primZone = (parsed.data.split("<ConfigItem key=\"primaryZone\" value=\"")[1].split("\"")[0] || lfrData.zoneId)
    const primCnty = (parsed.data.split("<ConfigItem key=\"primaryCounty\" value=\"")[1].split("\"")[0] || lfrData.cntyId)

    const id = Math.round(Math.random() * 1000000).toString()
      const cues = JSON.parse(fs.readFileSync(path.join(__dirname, "cues.json"), "utf-8"));
      const subs = clientSubscriptions.get(client.id) || new Set();
      let starType = ""
  try {


  for (const [cueId, cueObj] of Object.entries(cues)) {
    if (subs.has(cueObj.topic)) {
      // Add location info if not already there
      starType = String(cueObj.unitType).toUpperCase()
      cueObj.location = {display: `${lfrData.cityNm}, ${lfrData.stCd}`, primZone: primZone, primCnty: primCnty};
    }
  }

  fs.writeFileSync(path.join(__dirname, "cues.json"), JSON.stringify(cues, null, 2));
  console.log("Updated cues.json with location info.");
} catch (e) {
  console.error("Failed to update cues.json:", e);
}
    if(!(activeClients.has(clientIdSafe))) {
            const onlineEmbed = new EmbedBuilder()
        .setTitle("I2 Data Connection Started")
        .setDescription(`An i2${starType} for \`${lfrData.cityNm}, ${lfrData.stCd}\` has connected to STARSRV.\nThis connection was started at <t:${Math.round(new Date() / 1000)}>.\nConnection ID: \`${id}\``)
        .setFooter({text:"STARSRV US CENTRAL 1"})
        .setTimestamp()
        .setColor("Green")
      webhook.send({embeds:[onlineEmbed]})
    }
    activeClients.set(clientIdSafe, { filePath, lastSeen: Date.now(), location: lfrData, offline: false, id });
    // Check if subscribed to any cues


    console.log(`Saved XML from ${client.id} to ${filePath}`);

    updateLocations(parsed.data);
    return callback(null);
  }

    if (topic === 'connection/i1config') {
    const payload = packet.payload.toString();
    const sizeKB = Buffer.byteLength(payload) / 1024;

    if (sizeKB > 200)  {console.log("mpc received packet over 200kb"); };
    if (sizeKB > 200) {
      fs.writeFileSync(path.join(__dirname, "over200"), )
    }
    if (sizeKB > 200) {
      callback(new Error('Payload exceeds 200KB'))
    }


    const clientIdSafe = (client.id || 'client')
      .replace(/[\\/]/g, '-')
      .replace(/_/g, '-');

    const existingFile = [...activeClients.entries()].find(([id, data]) => id === clientIdSafe);
    if (existingFile) {
      try {
        fs.unlinkSync(existingFile[1].filePath);
      } catch {}
    }

    const filePath = path.join(uploadsDir_i1, `${Date.now()}_${clientIdSafe}.py`);
    fs.writeFileSync(filePath, payload);
    const location = payload.split("dsm.set('primaryCoopId','")[1].split("'")[0]

    const lfrLookup = await fetch(`http://192.168.0.134:3000/api/i2/l/coop/${location}?apiKey=4dc77e24801e8d21ea9ef72a9506ba0f`)
    const lfrData = await lfrLookup.json()
    const primZone = (payload.split("dsm.set('primaryZone','")[1].split("'")[0] || lfrData.zoneId)
    const primCnty = (payload.split("dsm.set('primaryCounty','")[1].split("'")[0] || lfrData.cntyId)
    const headendName = (payload.split("dsm.set('headendName','")[1].split("'")[0] || lfrData.cntyId)
    const mso = (payload.split("dsm.set('msoName','")[1].split("'")[0] || lfrData.cntyId)

    const id = Math.round(Math.random() * 1000000).toString()
      const cues = JSON.parse(fs.readFileSync(path.join(__dirname, "cues.json"), "utf-8"));
      const subs = clientSubscriptions.get(client.id) || new Set();
      let starType = ""

    if(!(activeClients.has(clientIdSafe))) {
            const onlineEmbed = new EmbedBuilder()
        .setTitle("I1 Data Connection Started")
        .setDescription(`An i1 for \`${lfrData.cityNm}, ${lfrData.stCd}\` has connected to STARSRV.\nThis connection was started at <t:${Math.round(new Date() / 1000)}>.\nConnection ID: \`${id}\`\nUnit: \`${headendName}\` - \`${mso}\``)
        .setFooter({text:"STARSRV US CENTRAL 1"})
        .setTimestamp()
        .setColor("Green")
      webhook.send({embeds:[onlineEmbed]})
    }
    activeClients.set(clientIdSafe, { filePath, lastSeen: Date.now(), location: lfrData, offline: false, id });
    // Check if subscribed to any cues


    console.log(`Saved PY CONFIG from ${client.id} to ${filePath}`);

    updateLocations_i1(payload);
    return callback(null);
  }

  if (client.secret === PUBLISH_SECRET) return callback(null);

  callback(new Error('Not authorized to publish to this topic'));
};

function updateLocations(xmlStr) {
  const parser = new XMLParser({
    ignoreAttributes: false,
    attributeNamePrefix: '',
  });

  let xml;
  try {
    xml = parser.parse(xmlStr);
  } catch (e) {
    console.error('Failed to parse XML:', e);
    return;
  }

  const keysToCheck = [
    'LocalRadarCity', 'MapCity', 'MetroMapCity', 'NearbyLocation',
    'RegionalMapCity', 'WinterGetawayLocation', 'TravelCity', 'SummerGetawayLocation'
  ];

  const keysToCheckForTides = [
    'TideStation'
  ];

  const currentGeneral = new Set();
  const currentAlerts = new Set();
  const currentTides = new Set();

  let configItems = [];
  try {
    const items = xml?.Config?.ConfigDef?.ConfigItems?.ConfigItem;
    if (!items) return;
    configItems = Array.isArray(items) ? items : [items];
  } catch {
    return;
  }

  for (const item of configItems) {
    const key = item.key || item['key'] || item['@_key'] || null;
    const value = item.value || item['value'] || item['@_value'] || null;
    if (!key || !value) continue;

    // General cities
    for (let i = 1; i <= 8; i++) {
      for (const baseKey of keysToCheck) {
        if (key === `${baseKey}${i}`) currentGeneral.add(value);
      }

      for (const baseKey of keysToCheckForTides) {
        if (key === `${baseKey}${i}`) currentTides.add(value);
      }
    }
    if (key === 'PrimaryLocation') currentGeneral.add(value);

    // Alert zones/counties
    if (key === 'primaryZone') currentAlerts.add(value);
    if (key === 'primaryCounty') currentAlerts.add(value);

    if (key === 'secondaryZones') {
      value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
    }

    if (key === 'secondaryCounties') {
      value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
    }
  }

  // Now collect from ALL active clients, not just current one
  for (const clientData of activeClients.values()) {
    const fileData = fs.readFileSync(clientData.filePath, 'utf-8');
    try {
      const parsedXml = parser.parse(fileData);
      const items = parsedXml?.Config?.ConfigDef?.ConfigItems?.ConfigItem;
      const allItems = Array.isArray(items) ? items : [items];

      for (const item of allItems) {
        const key = item.key || item['key'] || item['@_key'] || null;
        const value = item.value || item['value'] || item['@_value'] || null;
        if (!key || !value) continue;

        for (let i = 1; i <= 8; i++) {
          for (const baseKey of keysToCheck) {
            if (key === `${baseKey}${i}`) currentGeneral.add(value);
          }

          for (const baseKey of keysToCheckForTides) {
            if (key === `${baseKey}${i}`) currentTides.add(value);
          }
        }
        if (key === 'PrimaryLocation') currentGeneral.add(value);

        if (key === 'primaryZone') currentAlerts.add(value);
        if (key === 'primaryCounty') currentAlerts.add(value);

        if (key === 'secondaryZones') {
          value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
        }

        if (key === 'secondaryCounties') {
          value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
        }
      }
    } catch (e) {
      console.warn("Failed to parse active client's XML:", e.message);
    }
  }

  let locations = { general: [], alerts: [], tides: [], national: [] };
  try {
    locations = JSON.parse(fs.readFileSync(locationsFile, 'utf8'));
  } catch {}

  locations.general = [...currentGeneral].sort();
  locations.alerts = [...currentAlerts].sort();
  locations.tides = [...currentTides].sort();

  fs.writeFileSync(locationsFile, JSON.stringify(locations, null, 2));
  console.log('[✓] locations.json updated and cleaned');
}

function updateLocations_i1(pythonConfig) {
  const currentGeneral = JSON.parse(pythonConfig.split("wxdata.setInterestList('obsStation','1',")[1].split(")")[0].replace(",]", "]").replaceAll("'", "\""))
  const currentCoop = JSON.parse(pythonConfig.split("wxdata.setInterestList('coopId','1',")[1].split(")")[0].replace(",]", "]").replaceAll("'", "\""))

  let locations = { general: [], alerts: [], tides: [], national: [] };
  try {
    locations = JSON.parse(fs.readFileSync(locationsFile, 'utf8'));
  } catch {}

  locations.i1.obs = [...currentGeneral].sort();
  locations.i1.coop = [...currentCoop].sort();

  fs.writeFileSync(locationsFile, JSON.stringify(locations, null, 2));
  console.log('[✓] locations.json updated and cleaned');
}

/* setInterval(() => {
  activeClients.forEach((data, clientId) => {
    try {
      const payload = fs.readFileSync(data.filePath, 'utf8');
      const packet = {
        topic: 'connection/machineproductcfg',
        payload: JSON.stringify({ timestamp: Date.now(), data: payload })
      };
      aedes.publish(packet);
    } catch {}
  });
}, 5 * 60 * 1000); */


// Add this outside, at the top near `activeClients`

aedes.authorizeSubscribe = (client, sub, callback) => {
  const data = JSON.parse(fs.readFileSync(path.join(__dirname, "privateTopics.json"), "utf-8"));

  if (data[sub.topic] && !data[sub.topic].includes(client.secret)) {
    return callback(new Error("Not authorized for topic"));
  }

  const clientId = client.id;
  if (!clientSubscriptions.has(clientId)) clientSubscriptions.set(clientId, new Set());
  clientSubscriptions.get(clientId).add(sub.topic);

  callback(null, sub);
};


const listener = app.listen(MQTT_PORT, () => {
  console.log(`WebSocket MQTT at ws://localhost:${MQTT_PORT}${MQTT_PATH}`);
  setTimeout(() => {
    require("./data/i2/loop")()
    require("./data/i1/loop")()
    require("./data/i2/dataImages")("i2/rainwater-maps")
    require("./data/i2/radar")()
    require("./data/i1/radar")()
  }, 3 * 1000);
  require("./mqttlib").startCues()
  setInterval(() => require("./data/i2/radar")(), 2 * 60 * 1000)
  setInterval(() => require("./data/i1/radar")(), 5 * 60 * 1000)
  setInterval(() => require("./data/i2/dataImages")("i2/rainwater-maps"), 5 * 60 * 1000)

  const wss = new ws.Server({ noServer: true });

  listener.on('upgrade', (req, socket, head) => {
    if (req.url === MQTT_PATH) {
      wss.handleUpgrade(req, socket, head, (wsSocket) => {
        const stream = ws.createWebSocketStream(wsSocket);
        stream.on('error', err => {
          console.error('[WebSocket stream error]', err);
        });
        aedes.handle(stream);
      });
      
    } else {
      socket.destroy();
    }
  });
});

process.on('unhandledRejection', err => {
  console.log('[UNHANDLED REJECTION]', err);
});

process.on('uncaughtException', err => {
  console.log('[UNCAUGHT EXCEPTION]', err);
});


const { exec } = require("child_process");

function scheduleDailyRestart(hour = 3, minute = 30) {
  const now = new Date();
  const restartTime = new Date();

  restartTime.setHours(hour, minute, 0, 0);
  if (restartTime < now) restartTime.setDate(restartTime.getDate() + 1); // schedule for next day

  const delay = restartTime - now;
  console.log(`[!] MQTT restart scheduled in ${Math.round(delay / 1000 / 60)} minutes.`);

  setTimeout(() => {
    console.log("[!] Restarting MQTT broker...");

    // Example for PM2:
    exec("pm2 restart 14", (error, stdout, stderr) => {
      if (error) {
        console.error(`Error restarting: ${error.message}`);
        return;
      }
      console.log(`MQTT server restarted:\n${stdout}`);
    });

    // Reschedule again for next day
    scheduleDailyRestart(hour, minute);
  }, delay);
}

setInterval(() => {
  const mem = process.memoryUsage();
  console.log(`[Memory] RSS: ${(mem.rss / 1024 / 1024).toFixed(2)} MB, Heap: ${(mem.heapUsed / 1024 / 1024).toFixed(2)} MB`);
}, 60 * 60 * 1000);


scheduleDailyRestart(0, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(2, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(4, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(6, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(8, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(10, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(12, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(14, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(16, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(18, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(20, 0); // Restart daily at 3:30 AM
scheduleDailyRestart(22, 0); // Restart daily at 3:30 AM
