// saveAndSendLsImages.js
const fs = require("fs");
const path = require("path");
const fetch = require("node-fetch");
const zipLib = require("zip-lib");
const { exec, publishAsync, generateXML } = require("../../mqttlib");

const tempDir = path.join(__dirname, "temp");
if (!fs.existsSync(tempDir)) fs.mkdirSync(tempDir);

const baseAPI = "http://192.168.0.134:3000/api/i2/r/ls/latest";
const apiKey = "4dc77e24801e8d21ea9ef72a9506ba0f";

// Keys to download and send
const keys = [
  "breathing",
  "feelsLike",
  "windSpeed",
  "windSpeedGust",
  "grassPollen",
  "treePollen",
  "uv",
  "travel",
  "travel_1hr",
  "travel_12_1",
  "travel_12_2",
  "thunder",
  "ir",
  "vis",
  "wxscantravel",
  "waterTemps",
  "heat",
  "windChill",
  "sat_png"
];

async function saveAndSendLsImages(topic) {
  for (const key of keys) {
    try {
      const url = `${baseAPI}/${key}?apiKey=${apiKey}`;
      const res = await fetch(url);
      if (!res.ok) throw new Error(`Failed to fetch ${key}: ${res.status} ${res.statusText}`);

      const buffer = await res.buffer();
      const ext = ".png"; // Assume PNG
      const fileName = `${key}${ext}`;
      const filePath = path.join(tempDir, fileName);
      fs.writeFileSync(filePath, buffer);
      console.log(`✅ Saved ${key} -> ${filePath}`);

      await sendChangeset(filePath, fileName, topic);

    } catch (err) {
      console.error(`❌ Error with ${key}: ${err.message}`);
    }
  }
}

async function sendChangeset(filePath, fileName, topic) {
  await exec("resetAllStarBundleVersions(BType=Changeset)", topic);

  const vPath = `externalImage/domestic/rainwater/${fileName}`;
  const fileActions = [{ src: path.basename(filePath), dest: vPath, ogFileName: fileName }];

  const manifestXML = generateXML(fileActions, "Changeset");
  const metadataDir = path.join(tempDir, "MetaData");
  if (!fs.existsSync(metadataDir)) fs.mkdirSync(metadataDir);
  fs.writeFileSync(path.join(metadataDir, "manifest.xml"), manifestXML, "utf8");

  const zip = new zipLib.Zip();
  zip.addFile(filePath);
  zip.addFile(path.join(metadataDir, "manifest.xml"), "MetaData\\manifest.xml");

  const zipName = `${Math.floor(Math.random() * 100000000000)}_CS_StarBundle.zip`;
  const zipPath = path.join(tempDir, zipName);
  await zip.archive(zipPath);

  const fileBuffer = fs.readFileSync(zipPath);
  const base64Data = fileBuffer.toString("base64");

  const message = JSON.stringify({
    fileName: zipName,
    data: base64Data,
    workRequest: `stageStarBundle(File={filepath})`,
    payloadType: "i2StarBundle"
  });

  await publishAsync(topic, message);
  console.log(`📦 Changeset sent for ${fileName} as ${zipName}`);

  // Cleanup zip + manifest
  fs.unlinkSync(zipPath);
  fs.unlinkSync(path.join(metadataDir, "manifest.xml"));
}

// Export
module.exports = saveAndSendLsImages;
