// Run "npm i"

const {
    Client,
    GatewayIntentBits,
    ActivityType,
    EmbedBuilder,
    Events,
    WebhookClient
} = require('discord.js');
const config = require('./config.json');
const fs = require('fs');
const path = require('path');

// Import our library
const { createWeatherStarManager, Permissions } = require('../access');
const mqttlib = require('../mqttlib');

// Initialize Discord client
const client = new Client({
    intents: []
});

const webhook = new WebhookClient({
    url: "https://discord.com/api/webhooks/1401809185397538896/qs5X_2cnvxN4OgxTiybE7YzQHBGka1jRihiclE8UvhGXLoiMyiSaIWDlTBA9ZQGtWwxD"
});

// Initialize Weather Star Manager
let starManager;
const cuesPath = path.join(__dirname, '../cues.json');
const userstarsPath = path.join(__dirname, 'userstars.json');
const allowedUsersPath = path.join(__dirname, 'allowedUsers.json');
const configsPath = path.join(__dirname, 'configs.json');

// Load data
function loadStarManager() {
    try {
        const cuesData = JSON.parse(fs.readFileSync(cuesPath, 'utf-8'));
        const userstarsData = JSON.parse(fs.readFileSync(userstarsPath, 'utf-8'));
        starManager = createWeatherStarManager(cuesData, userstarsData);
        console.log('✅ Weather Star Manager loaded');
    } catch (e) {
        console.error('❌ Failed to load star manager:', e);
        starManager = createWeatherStarManager({}, {});
    }
}

// Save data
function saveStarManager() {
    try {
        const { cues, userStars } = starManager.exportToJSON();
        fs.writeFileSync(cuesPath, JSON.stringify(cues, null, 2));
        fs.writeFileSync(userstarsPath, JSON.stringify(userStars, null, 2));
        console.log('✅ Data saved successfully');
    } catch (e) {
        console.error('❌ Failed to save data:', e);
    }
}

// Load allowed users
function loadAllowedUsers() {
    try {
        return JSON.parse(fs.readFileSync(allowedUsersPath, 'utf-8'));
    } catch (e) {
        console.error('❌ Failed to load allowed users:', e);
        return [];
    }
}

// Utility: Parse relative time
function parseRelativeTime(input) {
    if (!input) return null;

    const timeUnits = {
        second: 1000, seconds: 1000, sec: 1000, secs: 1000, s: 1000,
        minute: 60 * 1000, minutes: 60 * 1000, min: 60 * 1000, mins: 60 * 1000, m: 60 * 1000,
        hour: 60 * 60 * 1000, hours: 60 * 60 * 1000, hr: 60 * 60 * 1000, hrs: 60 * 60 * 1000, h: 60 * 60 * 1000,
        day: 24 * 60 * 60 * 1000, days: 24 * 60 * 60 * 1000, d: 24 * 60 * 60 * 1000
    };

    const regex = /(\d+\.?\d*)\s*(\w+)/i;
    const match = input.trim().toLowerCase().match(regex);
    if (!match) return null;

    const value = parseFloat(match[1]);
    const unit = match[2];

    for (const [key, multiplier] of Object.entries(timeUnits)) {
        if (unit.startsWith(key)) {
            return value * multiplier;
        }
    }

    return null;
}

// Logging function
async function logMsg(interactor, interactStar, action) {
    try {
        const user = await client.users.fetch(interactor).catch(() => ({ username: 'Unknown User' }));
        const star = starManager.getUserStar(interactStar);
        const unit = starManager.getUnit(interactStar);

        const parts = [
            `SoftServe:`,
            `User ${interactor} (**${user.username}**) has ${action} to STAR ${interactStar}`
        ];

        if (star) {
            const ownerUser = await client.users.fetch(star.owner).catch(() => ({ username: 'Unknown Owner' }));
            parts.push(`(${ownerUser.username}/${star.owner})`);
        }

        if (unit) {
            const unitType = unit.unitType.toUpperCase();
            const location = unit.location ? unit.location.display : 'Unknown Location';
            parts.push(`which is a ${unitType} for \`${location}\``);
        }

        await webhook.send({ content: parts.join(' ') });
    } catch (e) {
        console.error('Failed to send webhook message:', e);
    }
}

// Permission mapping for commands
const PERMISSION_MAP = {
    auth: { add: Permissions.OWNER, remove: Permissions.OWNER, delete: Permissions.OWNER },
    playlist: { loadrun: Permissions.CUES, cancel: Permissions.CUES },
    restart: { system: Permissions.OWNER, i2service: Permissions.MANAGEMENT, process: Permissions.MANAGEMENT },
    cuer: { toggle: Permissions.CUES, force: Permissions.CUES, configure: Permissions.CUES, manual: Permissions.CUES },
    'fake-alert': Permissions.CUES,
    'change-config': Permissions.MANAGEMENT,
    'exec': Permissions.OWNER
};

// Get required permission for command
function getRequiredPermission(commandName, subcommandName = null) {
    const perm = PERMISSION_MAP[commandName];
    if (!perm) return null;
    if (typeof perm === 'string') return perm;
    if (subcommandName && perm[subcommandName]) return perm[subcommandName];
    return null;
}

// Bot ready event
client.on('ready', () => {
    console.log('[bot] ready');
    client.user.setActivity(config.activity, { type: ActivityType.Watching });
    webhook.send({ content: 'SoftServe is now online.' });
    loadStarManager();
});

// Autocomplete handler
client.on(Events.InteractionCreate, async (interaction) => {
    if (!interaction.isAutocomplete()) return;

    const focusedOption = interaction.options.getFocused(true);
    const commandName = interaction.commandName;
    const subcommandName = interaction.options.getSubcommand(false);
    const userId = interaction.user.id;
    const starId = interaction.options.getString('star');
    const unit = starId ? starManager.getUnit(starId) : null;
    const unitType = unit?.unitType?.toLowerCase();
    const results = [];

    if (focusedOption.name === 'star') {
        const requiredPermission = getRequiredPermission(commandName, subcommandName);
        const options = starManager.getAutocompleteOptions(userId, requiredPermission);
        
        results.push(...options.filter(opt => 
            opt.name.toLowerCase().includes(focusedOption.value.toLowerCase())
        ));

    } else if (['flavor', 'ldl', 'sidebar'].includes(focusedOption.name)) {
        const flavorMap = {
            flavor: {
                jr: [
                    ['Enhanced', 'domestic/V'],
                    ['Nemo', 'domestic/N'],
                    ['Azul', 'domestic/azul i2 jr'],
                    ['Enhanced Breaking', 'domestic/V1'],
                    ['Enhanced Severe', 'domestic/V2']
                ],
                xd: [
                    ['Nemo', 'domestic/V'],
                    ['Enhanced', 'domestic/V2016'],
                    ['Azul', 'domestic/Azul'],
                    ['Nemo Breaking', 'domestic/V1'],
                    ['Nemo Severe', 'domestic/V2']
                ],
                global: [
                    ['Nemo xD/Enhanced Jr', 'domestic/V'],
                    ['Nemo Jr', 'domestic/N'],
                    ['Enhanced xD', 'domestic/V2016'],
                    ['Azul Jr', 'domestic/azul i2 jr'],
                    ['Azul xD', 'domestic/Azul']
                ]
            },
            ldl: {
                jr: [
                    ['Enhanced', 'domestic/ldlC'],
                    ['Nemo', 'domestic/ldlE'],
                    ['Azul', 'domestic/azulldl_16'],
                    ['Enhanced Breaking', 'domestic/ldlD'],
                    ['Nemo Breaking', 'domestic/ldlF']
                ],
                xd: [
                    ['Enhanced', 'domestic/ldlC'],
                    ['Nemo', 'domestic/ldlE'],
                    ['Azul', 'domestic/azulldl_16'],
                    ['Azul Large', 'domestic/azulldl'],
                    ['Enhanced Breaking', 'domestic/ldlD'],
                    ['Nemo Breaking', 'domestic/ldlF']
                ],
                global: [
                    ['Enhanced', 'domestic/ldlC'],
                    ['Nemo', 'domestic/ldlE'],
                    ['Azul', 'domestic/azulldl_16'],
                    ['Azul Large', 'domestic/azulldl'],
                    ['Enhanced Breaking', 'domestic/ldlD'],
                    ['Nemo Breaking', 'domestic/ldlF']
                ]
            },
            sidebar: {
                jr: [['None', 'none']],
                xd: [
                    ['Enhanced Sidebar (1)', 'domestic/sidebarXC'],
                    ['Enhanced Sidebar (2)', 'domestic/sidebarC'],
                    ['Enhanced Sidebar Breaking (1)', 'domestic/sidebarXD'],
                    ['Enhanced Sidebar Breaking (2)', 'domestic/sidebarD'],
                    ['Nemo Sidebar', 'domestic/sidebarE'],
                    ['None', 'none']
                ],
                global: [
                    ['Enhanced Sidebar (1)', 'domestic/sidebarXC'],
                    ['Enhanced Sidebar (2)', 'domestic/sidebarC'],
                    ['Enhanced Sidebar Breaking (1)', 'domestic/sidebarXD'],
                    ['Enhanced Sidebar Breaking (2)', 'domestic/sidebarD'],
                    ['Nemo Sidebar', 'domestic/sidebarE'],
                    ['None', 'none']
                ]
            }
        };

        const category = flavorMap[focusedOption.name];
        const list = unitType === 'jr' ? category.jr : unitType === 'xd' ? category.xd : category.global;

        const filtered = list
            .filter(([name]) => name.toLowerCase().includes(focusedOption.value.toLowerCase()))
            .map(([name, value]) => ({ name: `${name} (${value})`, value }));

        results.push(...filtered);

    } else if (focusedOption.name === 'config') {
        const configs = JSON.parse(fs.readFileSync(configsPath, 'utf8'));
        const filtered = configs
            .filter(config =>
                `${config.city}, ${config.state}`.toLowerCase().includes(focusedOption.value.toLowerCase()) ||
                config.isp?.toLowerCase().includes(focusedOption.value.toLowerCase())
            )
            .map(config => ({
                name: `${config.city.toUpperCase()}, ${config.state} - ${config.isp}`,
                value: config.filePath
            }));

        results.push(...filtered);

    } else if (focusedOption.name === 'background') {
        let backgroundsPath = unitType === 'jr' 
            ? path.join(__dirname, '../jr.json')
            : path.join(__dirname, '../xd.json');

        try {
            const backgrounds = JSON.parse(fs.readFileSync(backgroundsPath, 'utf8'));
            const filtered = backgrounds
                .filter(bg => bg.Description.toLowerCase().includes(focusedOption.value.toLowerCase()))
                .map(bg => ({
                    name: `${bg.Description} (domesticAds/TAG${bg.ID})`,
                    value: `domesticAds/TAG${bg.ID}`
                }));
            results.push(...filtered);
        } catch (e) {
            console.error('Failed to read backgrounds file:', e);
        }
    }

    try {
        await interaction.respond(results.slice(0, 25));
    } catch (e) {
        console.error('Failed to respond to autocomplete:', e);
    }
});

// Command handler
client.on(Events.InteractionCreate, async (interaction) => {
    if (!interaction.isChatInputCommand()) return;

    // Check if user is allowed
    const allowList = loadAllowedUsers();
    if (!allowList.includes(interaction.user.id)) {
        return interaction.reply({
            ephemeral: true,
            content: 'This bot is only usable by Rainwater authenticated users!'
        });
    }

    await interaction.deferReply();

    try {
        await handleCommand(interaction);
    } catch (error) {
        console.error('Command error:', error);
        await interaction.editReply({
            content: 'An error occurred while processing your command.',
            ephemeral: true
        });
    }
});

// Main command handler
async function handleCommand(interaction) {
    const command = interaction.commandName;
    
    switch (command) {
        case 'register':
            await handleRegister(interaction);
            break;
        case 'playlist':
            await handlePlaylist(interaction);
            break;
        case 'restart':
            await handleRestart(interaction);
            break;
        case 'cuer':
            await handleCuer(interaction);
            break;
        case 'auth':
            await handleAuth(interaction);
            break;
        case 'fake-alert':
            await handleFakeAlert(interaction);
            break;
        case 'change-config':
            await handleChangeConfig(interaction);
            break;
        case 'location-info':
            await handleLocationInfo(interaction);
            break;
        case 'exec':
            await handleExec(interaction);
            break;
        default:
            await interaction.editReply({ content: 'Unknown command.', ephemeral: true });
    }
}

// Check access helper
function checkAccess(starId, userId, permission) {
    return starManager.canUserAccessUnit(userId, starId, permission);
}

// Register command
async function handleRegister(interaction) {
    const unitType = interaction.options.getString('unit-type');
    const nickname = interaction.options.getString('nickname', false);
    
    const unitId = starManager.generateUnitId();
    const topic = `i2/${unitId}`;

    const config = {
        topic,
        unitType,
        nickname: nickname || '',
        flavor: 'domestic/V',
        duration: 1950,
        timing: { every: 10, onthe: 8 },
        background: 9,
        ldl: 'domestic/ldlE',
        enabled: false,
        sidebar: { enabled: false, flavor: 'domestic/sidebarE' }
    };

    starManager.addUnit(unitId, config);
    starManager.createUserStar(unitId, interaction.user.id);
    saveStarManager();

    const embed = new EmbedBuilder()
        .setTitle('Successfully registered new I2 unit!')
        .setDescription(`Registered a \`${unitType}\` unit, you can now **grant access** & **use this with the bot**.\n\nPlease **subscribe to the topic** \`${topic}\` on the **encoder**.`)
        .setColor('Green')
        .setTimestamp()
        .setFooter({ text: 'Brought to you by Rainwater.' });

    await logMsg(interaction.user.id, unitId, 'registered an I2');
    await interaction.editReply({ embeds: [embed], ephemeral: true });
}

// Playlist command
async function handlePlaylist(interaction) {
    const starId = interaction.options.getString('star');
    const subcommand = interaction.options.getSubcommand(true);

    if (!checkAccess(starId, interaction.user.id, Permissions.CUES)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const unit = starManager.getUnit(starId);
    if (!unit) {
        return interaction.editReply({ content: "Unit not found!", ephemeral: true });
    }

    if (subcommand === 'loadrun') {
        const flavor = interaction.options.getString('flavor');
        const duration = interaction.options.getInteger('duration');
        const background = interaction.options.getString('background', false);

        mqttlib.sendI2Playlist(unit.topic, flavor, duration, 4, background, 15);

        const embed = new EmbedBuilder()
            .setTitle('Successfully sent playlist!')
            .setDescription(`**Successfully sent a playlist** for **I2** for \`${unit.location?.display || 'Unknown'}\` with the following arguments:\n**Flavor**: \`${flavor}\`\n**Duration**: \`${duration}\`\n**Logo/Background**: \`${background || 'generic'}\`\nPlease wait **15** seconds for your playlist to run.`)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'sent a playlist');
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'cancel') {
        const presId = interaction.options.getString('presentation-id', true);
        mqttlib.exec(`cancelPres(PresentationId=${presId})`, unit.topic);

        const embed = new EmbedBuilder()
            .setTitle('Successfully cancelled playlist!')
            .setDescription(`Successfully **cancelled the playlist** for the selected STAR with pres ID \`${presId}\``)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'cancelled a playlist');
        return interaction.editReply({ embeds: [embed] });
    }
}

// Restart command
async function handleRestart(interaction) {
    const starId = interaction.options.getString('star');
    const subcommand = interaction.options.getSubcommand(true);

    const requiredPerm = PERMISSION_MAP.restart[subcommand];
    if (!checkAccess(starId, interaction.user.id, requiredPerm)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const unit = starManager.getUnit(starId);
    if (!unit) {
        return interaction.editReply({ content: "Unit not found!", ephemeral: true });
    }

    let title, description, logAction;

    if (subcommand === 'system') {
        mqttlib.exec('rebootStar(CommandId=Rainwater_Bot)', unit.topic);
        title = 'Successfully restarted system!';
        description = `Successfully **restarted** the STAR unit for \`${unit.location?.display || 'Unknown'}\`!`;
        logAction = 'rebooted';

    } else if (subcommand === 'i2service') {
        mqttlib.exec('restartI2Service(CommandId=Rainwater_Bot)', unit.topic);
        title = 'Successfully restarted I2 Service!';
        description = `Successfully **restarted I2 Service** on the STAR unit for \`${unit.location?.display || 'Unknown'}\`!`;
        logAction = 'restarted i2 service';

    } else if (subcommand === 'process') {
        const processName = interaction.options.getString('process');
        mqttlib.exec(`restartProcess(ProcessName=${processName})`, unit.topic);
        title = 'Successfully restarted an I2 process!';
        description = `Successfully **restarted an I2 process** on the STAR unit for \`${unit.location?.display || 'Unknown'}\`!`;
        logAction = 'restarted an i2 process';
    }

    const embed = new EmbedBuilder()
        .setTitle(title)
        .setDescription(description)
        .setColor('Green')
        .setTimestamp();

    await logMsg(interaction.user.id, starId, logAction);
    return interaction.editReply({ embeds: [embed] });
}

// Cuer command
async function handleCuer(interaction) {
    const starId = interaction.options.getString('star');
    const subcommand = interaction.options.getSubcommand(true);

    if (!checkAccess(starId, interaction.user.id, Permissions.CUES)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const unit = starManager.getUnit(starId);
    if (!unit) {
        return interaction.editReply({ content: "Unit not found!", ephemeral: true });
    }

    if (subcommand === 'toggle') {
        const newState = unit.toggle();
        saveStarManager();

        const embed = new EmbedBuilder()
            .setTitle('Successfully toggled automatic cueing!')
            .setDescription(`**Automatic cueing** for **I2** for \`${unit.location?.display || 'Unknown'}\` is now set to: \`${newState}\``)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'toggled cueing for');
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'force') {
        const bg = unit.getBackgroundString();
        const startFlavors = unit.getStartFlavors();

        await mqttlib.sendI2Playlist(
            unit.topic,
            unit.flavor,
            unit.duration,
            4,
            bg,
            20,
            [{ id: 'ldl3' }, { id: 'sidebar2' }, { id: 'ldl1' }],
            startFlavors
        );

        const embed = new EmbedBuilder()
            .setTitle('Successfully forced an automatic cue!')
            .setDescription(`**Playlist parameters** for **I2** for \`${unit.location?.display || 'Unknown'}\`:\n**Flavor**: \`${unit.flavor}\`\n**Duration**: \`${unit.duration}\`\n**Background**: \`${bg || 'generic'}\``)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'forced a cue for');
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'configure') {
        const updates = {};

        if (interaction.options.getString('flavor', false)) {
            updates.flavor = interaction.options.getString('flavor');
        }
        if (interaction.options.getString('timing', false)) {
            updates.timing = interaction.options.getString('timing');
        }
        if (interaction.options.getInteger('duration', false)) {
            updates.duration = interaction.options.getInteger('duration');
        }
        if (interaction.options.getBoolean('enabled', false) !== null) {
            updates.enabled = interaction.options.getBoolean('enabled');
        }
        if (interaction.options.getString('ldl', false)) {
            updates.ldl = interaction.options.getString('ldl');
        }
        if (interaction.options.getString('sidebar', false)) {
            updates.sidebar = interaction.options.getString('sidebar');
        }
        if (interaction.options.getString('background', false)) {
            updates.background = interaction.options.getString('background');
        }

        starManager.updateUnit(starId, updates);
        saveStarManager();

        const bg = unit.getBackgroundString();

        const embed = new EmbedBuilder()
            .setTitle('Successfully configured automatic cueing!')
            .setDescription(`**Playlist parameters** for **I2** for \`${unit.location?.display || 'Unknown'}\`:\n**Enabled**: \`${unit.enabled}\`\n**Flavor**: \`${unit.flavor}\`\n**Duration**: \`${unit.duration}\`\n**Background**: \`${bg || 'generic'}\`\n**LDL**: \`${unit.ldl}\`\n**Timing**: \`${unit.timing}\`\n**Sidebar**: \`${unit.sidebar.flavor}, ${unit.sidebar.enabled}\``)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'configured cueing for');
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'manual') {
        const flavor = interaction.options.getString('flavor');
        const duration = interaction.options.getInteger('duration');
        const background = interaction.options.getString('background', false);
        const startFlavors = unit.getStartFlavors();

        mqttlib.sendI2Playlist(
            unit.topic,
            flavor,
            duration,
            4,
            background,
            15,
            [{ id: 'ldl3' }, { id: 'sidebar2' }],
            startFlavors
        );

        const embed = new EmbedBuilder()
            .setTitle('Successfully sent playlist!')
            .setDescription(`**Successfully sent a playlist** for **I2** for \`${unit.location?.display || 'Unknown'}\` with the following arguments:\n**Flavor**: \`${flavor}\`\n**Duration**: \`${duration}\`\n**Logo/Background**: \`${background || 'generic'}\`\nPlease wait **15** seconds for your playlist to run.`)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'sent a playlist');
        return interaction.editReply({ embeds: [embed] });
    }
}

// Auth command
async function handleAuth(interaction) {
    const starId = interaction.options.getString('star');
    const subcommand = interaction.options.getSubcommand(true);

    if (!checkAccess(starId, interaction.user.id, Permissions.OWNER)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const user = interaction.options.getUser('user');

    if (subcommand === 'add') {
        const permissions = interaction.options.getString('permissions')
            .toLowerCase()
            .split(',')
            .map(p => p.trim());

        starManager.grantUserAccess(starId, user.id, permissions);
        saveStarManager();

        const embed = new EmbedBuilder()
            .setTitle('Successfully granted access!')
            .setDescription(`${user} now **has access** to **your I2** with the following **permissions**:\n\`${permissions.join(', ')}\``)
            .setColor('Green')
            .setTimestamp()
            .setFooter({ text: 'Brought to you by Rainwater.' });

        await logMsg(interaction.user.id, starId, `authed a user ${user.id}`);
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'remove') {
        starManager.revokeUserAccess(starId, user.id);
        saveStarManager();

        const embed = new EmbedBuilder()
            .setTitle('Successfully removed access!')
            .setDescription(`${user} **no longer** has access to your I2!`)
            .setColor('Green')
            .setTimestamp()
            .setFooter({ text: 'Brought to you by Rainwater.' });

        await logMsg(interaction.user.id, starId, `unauthed a user ${user.id}`);
        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'delete') {
        starManager.removeUnit(starId);
        saveStarManager();

        const embed = new EmbedBuilder()
            .setTitle('Successfully removed unit!')
            .setDescription(`The provided unit is **no longer in our database**.`)
            .setColor('Green')
            .setTimestamp();

        await logMsg(interaction.user.id, starId, 'deleted i2 from db');
        return interaction.editReply({ embeds: [embed] });
    }
}

// Fake alert command
async function handleFakeAlert(interaction) {
    const starId = interaction.options.getString('star');

    if (!checkAccess(starId, interaction.user.id, Permissions.CUES)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const type = interaction.options.getString('type', false) || 'generic';
    const name = interaction.options.getString('name');
    const bulletin = interaction.options.getString('bulletin');
    const expiration = interaction.options.getString('expiration');

    const expirationTime = Date.now() + parseRelativeTime(expiration);
    mqttlib.sendFakeAlert(type, name, bulletin, 'An alert remains in effect.', starId, expirationTime);

    const bulletinTruncated = bulletin.length > 250 
        ? 'truncated due to length, please check star/interaction options' 
        : bulletin;

    const embed = new EmbedBuilder()
        .setTitle('Successfully sent fake alert!')
        .setDescription(`Successfully sent your alert to the I2. Keep your eyes peeled!\n**Alert Details**\n**Name**: \`${name}\`\n**Bulletin Message**: \`${bulletinTruncated}\`\n**Expires In**: \`${expiration}\``)
        .setColor('Green')
        .setFooter({ text: 'Brought to you by Rainwater.' })
        .setTimestamp();

    await logMsg(interaction.user.id, starId, 'sent a fake alert');
    return interaction.editReply({ embeds: [embed] });
}

// Change config command
async function handleChangeConfig(interaction) {
    const starId = interaction.options.getString('star');

    if (!checkAccess(starId, interaction.user.id, Permissions.MANAGEMENT)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const unit = starManager.getUnit(starId);
    if (!unit) {
        return interaction.editReply({ content: "Unit not found!", ephemeral: true });
    }

    const configOption = interaction.options.get('config')?.value;
    if (!configOption) {
        return interaction.editReply({ ephemeral: true, content: 'No config selected.' });
    }

    const relativeConfigPath = configOption.replace(/--/g, '\\\\');
    const normalizedPath = path.normalize(relativeConfigPath);
    const fullConfigPath = path.join(__dirname, 'configs', normalizedPath.replace(/^configs[\\/]+/, '').replaceAll('\\', '/'));

    if (!fullConfigPath.startsWith(path.join(__dirname, 'configs'))) {
        return interaction.editReply({ ephemeral: true, content: 'Invalid config path. Access denied.' });
    }

    if (!fs.existsSync(fullConfigPath)) {
        return interaction.editReply({ ephemeral: true, content: "This config doesn't exist locally!" });
    }

    const fileContent = fs.readFileSync(fullConfigPath, 'utf-8');
    mqttlib.sendMachineProductCfg(fileContent, unit.topic);

    const configsData = JSON.parse(fs.readFileSync(configsPath, 'utf8'));
    const matchedConfig = configsData.find(c => c.filePath === relativeConfigPath);

    const embed = new EmbedBuilder()
        .setTitle('Successfully sent I2 Config!')
        .setDescription(
            matchedConfig
                ? `Your I2 should now have a config for \`${matchedConfig.city}, ${matchedConfig.state}\`!`
                : 'Your I2 should now have the selected config!'
        )
        .setColor('Green')
        .setFooter({ text: 'Brought to you by Rainwater.' })
        .setTimestamp();

    await logMsg(interaction.user.id, starId, 'sent a config to unit');
    return interaction.editReply({ embeds: [embed] });
}

// Location info command
async function handleLocationInfo(interaction) {
    const subcommand = interaction.options.getSubcommand();
    const wistApiKey = '4dc77e24801e8d21ea9ef72a9506ba0f';

    if (subcommand === 'location-id') {
        const locId = interaction.options.getString('id', true);
        const apiReq = await fetch(`https://wist.minnwx.com/api/i2/l/lid/${locId}?apiKey=${wistApiKey}`);

        if (!apiReq.ok) {
            return interaction.editReply({ ephemeral: true, content: "That location ID doesn't exist, or the API request failed!" });
        }

        const apiRes = await apiReq.json();
        if (!apiRes.cityNm) {
            return interaction.editReply({ ephemeral: true, content: "That city doesn't exist." });
        }

        const embed = new EmbedBuilder()
            .setTitle('I2 LocId Lookup')
            .setDescription(`**Location ID Lookup** for \`${locId}\`\n\n**City Name**: \`${apiRes.cityNm}\`\n**Zone ID**: \`${apiRes.zoneId}\`\n**County ID**: \`${apiRes.cntyId}\`\n**Lat/Lon**: \`${apiRes.lat}\`/\`${apiRes.long}\`\n**ZIP**: \`${apiRes.zip2locId || 'No zip2locId Found'}\`\n**Tecci ID**: \`${apiRes.primTecci}\`\n**Coop ID**: \`${apiRes.coopId}\``)
            .setColor('Blue')
            .setFooter({ text: 'Brought to you by Rainwater Multimedia!' });

        return interaction.editReply({ embeds: [embed] });

    } else if (subcommand === 'name') {
        const city = interaction.options.getString('city', true);
        const state = interaction.options.getString('state', true);
        const apiReq = await fetch(`https://wist.minnwx.com/api/i2/l/name/${city}/${state}?apiKey=${wistApiKey}`);

        if (!apiReq.ok) {
            return interaction.editReply({ ephemeral: true, content: "That location doesn't exist in the LFRecord, or the API request failed!" });
        }

        const apiRes = await apiReq.json();
        if (!apiRes.cityNm) {
            return interaction.editReply({ ephemeral: true, content: "That city doesn't exist." });
        }

        const embed = new EmbedBuilder()
            .setTitle('I2 Location Lookup')
            .setDescription(`**Location Lookup** for \`${city.toUpperCase()}, ${state.toUpperCase()}\`\n\n**LocID**: \`${apiRes.locType}_${apiRes.cntryCd}_${apiRes.locId}\`\n**City Name**: \`${apiRes.cityNm}\`\n**Zone ID**: \`${apiRes.zoneId}\`\n**County ID**: \`${apiRes.cntyId}\`\n**Lat/Lon**: \`${apiRes.lat}\`/\`${apiRes.long}\`\n**ZIP**: \`${apiRes.zip2locId || 'No zip2locId Found'}\`\n**Tecci ID**: \`${apiRes.primTecci}\`\n**Coop ID**: \`${apiRes.coopId}\``)
            .setColor('Blue')
            .setFooter({ text: 'Brought to you by Rainwater Multimedia!' });

        return interaction.editReply({ embeds: [embed] });
    }
}

// Exec command
async function handleExec(interaction) {
    const starId = interaction.options.getString('star');

    if (!checkAccess(starId, interaction.user.id, Permissions.OWNER)) {
        return interaction.editReply({ content: "You don't have access to this STAR!", ephemeral: true });
    }

    const unit = starManager.getUnit(starId);
    if (!unit) {
        return interaction.editReply({ content: "Unit not found!", ephemeral: true });
    }

    const command = interaction.options.get('command')?.value;
    mqttlib.exec(command, unit.topic);

    const embed = new EmbedBuilder()
        .setTitle('Successfully ran I2 Exec command!')
        .setDescription(`Command sent to the I2:\n \`${command}\``)
        .setColor('Green')
        .setFooter({ text: 'Brought to you by Rainwater.' })
        .setTimestamp();

    await logMsg(interaction.user.id, starId, 'sent a command to unit');
    return interaction.editReply({ embeds: [embed] });
}

// Log in to the bot
client.login(config.token);