const express = require('express');
const fs = require('fs');
const path = require('path');
const aedes = require('aedes')();
const ws = require('ws');
const axios = require('axios');
const { exec } = require('child_process');
const config = require('../config.json');

const lockpath = path.join(__dirname, 'server.lock');
const MQTT_PATH = '/mqtt';
const MQTT_PORT = config.encoderPort;
const AUTH_URL = `${config.wistAddress}/api/auth`;
const PUBLISH_SECRET = config.mainWistKey;

// ============== PROCESS LOCK ==============
function isProcessRunning(pid) {
  try {
    process.kill(pid, 0);
    return true;
  } catch (e) {
    return false;
  }
}

if (fs.existsSync(lockpath)) {
  try {
    const existingPid = parseInt(fs.readFileSync(lockpath, 'utf-8'), 10);
    if (!isNaN(existingPid) && isProcessRunning(existingPid)) {
      console.error(`Another instance (PID ${existingPid}) is already running.`);
      exec("pkill \"node /root/rain\"");
      process.exit(1);
    } else {
      console.warn('Stale lock file found. Removing.');
      fs.unlinkSync(lockpath);
    }
  } catch (e) {
    console.error('Failed to read lock file:', e);
    process.exit(1);
  }
}

fs.writeFileSync(lockpath, process.pid.toString(), 'utf-8');

const cleanUp = () => {
  if (fs.existsSync(lockpath)) {
    const pidInFile = parseInt(fs.readFileSync(lockpath, 'utf-8'), 10);
    if (pidInFile === process.pid) {
      fs.unlinkSync(lockpath);
    }
  }
  process.exit();
};

process.on('SIGINT', cleanUp);
process.on('SIGTERM', cleanUp);
process.on('SIGQUIT', cleanUp);
process.on('exit', cleanUp);
process.on('uncaughtException', (err) => {
  console.error('Uncaught exception:', err);
  cleanUp();
});

// ============== EXPRESS APP ==============
const app = express();
app.use(express.json());

const clientSubscriptions = new Map();

async function useApiKey(req, res, next) {
  if (req.query.apiKey) {
    try {
      const result = await require("../auth/base/encoder-auth")(req.query.apiKey)
      if (result === true) {
        return next();
      }
    } catch (e) {
      console.error('Auth check failed:', e.message);
    }
  }
  return res.status(404).send("Unauthorized.");
}

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ 
    status: 'ok', 
    uptime: process.uptime(),
    memory: process.memoryUsage(),
    timestamp: Date.now()
  });
});

app.get("/api/version", useApiKey, (req, res) => {
  const pkg = JSON.parse(fs.readFileSync(path.join(__dirname, "package.json"), "utf-8"));
  res.json({ latest: pkg.version, srv: pkg.serverVersion, i1: pkg.i1Version });
});

app.get("/api/latest/encoder.exe", useApiKey, (req, res) => {
  res.sendFile(path.join(__dirname, "rainwater-encoder.exe"));
});

app.get("/api/latest/updater.exe", useApiKey, (req, res) => {
  res.sendFile(path.join(__dirname, "rainwater-updater.exe"));
});

// ============== MQTT AUTHENTICATION ==============
aedes.authenticate = async (client, username, password, callback) => {
  const secret = username?.toString();
  if (!secret) return callback(new Error('No secret provided'), false);

  const handlers = require('./handlers');
  return handlers.authorize(client, username, password, callback);
};

// ============== MQTT AUTHORIZATION ==============
aedes.authorizePublish = async (client, packet, callback) => {
  if (client.secret === PUBLISH_SECRET) return callback(null);
  
  // Delegate to handlers module
  const handlers = require('./handlers');
  return handlers.handlePublish(client, packet, callback, clientSubscriptions);
};

aedes.authorizeSubscribe = (client, sub, callback) => {
  const clientId = client.id;
  if (!clientSubscriptions.has(clientId)) clientSubscriptions.set(clientId, new Set());
  clientSubscriptions.get(clientId).add(sub.topic);
  callback(null, sub);
};

// ============== CLIENT DISCONNECT ==============
aedes.on('clientDisconnect', (client) => {
  clientSubscriptions.delete(client.id);
  const handlers = require('./handlers');
  handlers.handleClientDisconnect(client);
});

// ============== START SERVER ==============
const listener = app.listen(MQTT_PORT, () => {
  console.log(`WebSocket MQTT at ws://localhost:${MQTT_PORT}${MQTT_PATH}`);
  const wss = new ws.Server({ noServer: true });

  listener.on('upgrade', (req, socket, head) => {
    if (req.url === MQTT_PATH) {
      wss.handleUpgrade(req, socket, head, (wsSocket) => {
        const stream = ws.createWebSocketStream(wsSocket);
        
        const timeout = setTimeout(() => {
          if (!wsSocket.isAlive) {
            console.warn('[WebSocket] Closing idle connection after 30 min timeout');
            stream.destroy();
          }
        }, 30 * 60 * 1000);
        
        wsSocket.isAlive = true;
        wsSocket.on('pong', () => { 
          wsSocket.isAlive = true; 
          clearTimeout(timeout); 
        });
        
        stream.on('error', err => {
          console.error('[WebSocket stream error]', err.message);
          clearTimeout(timeout);
        });
        
        stream.on('end', () => clearTimeout(timeout));
        stream.on('close', () => clearTimeout(timeout));
        
        aedes.handle(stream);
      });
    } else {
      socket.destroy();
    }
  });
});

// ============== HEALTH MONITORING ==============
const HEALTH_CHECK_INTERVAL = 60 * 1000; // Check every minute
const HEALTH_CHECK_TIMEOUT = 5000; // 5 second timeout
let consecutiveFailures = 0;
const MAX_FAILURES = 3;

async function checkServerHealth() {
  try {
    const response = await axios.get(`http://localhost:${MQTT_PORT}/health`, {
      timeout: HEALTH_CHECK_TIMEOUT
    });
    
    if (response.data.status === 'ok') {
      consecutiveFailures = 0;
      return true;
    }
  } catch (error) {
    console.error(`[Health Check] Failed: ${error.message}`);
    consecutiveFailures++;
    
    if (consecutiveFailures >= MAX_FAILURES) {
      console.error(`[!!!] CRITICAL - Server unreachable after ${MAX_FAILURES} attempts. Restarting...`);
      exec("pm2 restart 2", (err, stdout, stderr) => {
        if (err) {
          console.error(`Error restarting: ${err.message}`);
          process.exit(1); // Force exit if PM2 restart fails
        }
        console.log(`Server restarted:\n${stdout}`);
      });
    }
  }
  return false;
}

setInterval(checkServerHealth, HEALTH_CHECK_INTERVAL);

// ============== ERROR HANDLERS ==============
process.on('unhandledRejection', err => {
  console.error('[UNHANDLED REJECTION]', err);
});

process.on('uncaughtException', err => {
  console.error('[UNCAUGHT EXCEPTION]', err);
});

console.log('[✓] MQTT Server started successfully');