const fs = require('fs');
const path = require('path');
const { XMLParser } = require('fast-xml-parser');
const { WebhookClient, EmbedBuilder } = require('discord.js');
const config = require('../config.json');

const uploadsDir = path.join(__dirname, 'uploads');
const uploadsDir_i1 = path.join(__dirname, 'uploads_i1');
const locationsFile = path.join(__dirname, 'locations.json');

if (!fs.existsSync(uploadsDir)) fs.mkdirSync(uploadsDir);
if (!fs.existsSync(uploadsDir_i1)) fs.mkdirSync(uploadsDir_i1);

const webhook = new WebhookClient({ 
  url: config.webhook
});

const activeClients = new Map();

// ============== FILE CLEANUP ==============
async function cleanupOldFiles(dir, maxAgeMs) {
  try {
    const files = await fs.promises.readdir(dir);
    const now = Date.now();
    const batch = 50;
    
    for (let i = 0; i < files.length; i += batch) {
      await Promise.all(
        files.slice(i, i + batch).map(async (f) => {
          try {
            const timestamp = parseInt(f.split("_")[0]);
            if (now - timestamp > maxAgeMs) {
              await fs.promises.unlink(path.join(dir, f));
            }
          } catch (e) {
            // Silently skip
          }
        })
      );
    }
  } catch (e) {
    console.error(`Cleanup failed for ${dir}:`, e.message);
  }
}

// Initial cleanup
cleanupOldFiles(uploadsDir, 2 * 60 * 60 * 1000);
cleanupOldFiles(path.join(__dirname, "../public/encoder"), 2 * 60 * 60 * 1000);
cleanupOldFiles(uploadsDir_i1, 2 * 60 * 60 * 1000);

// ============== CLIENT CLEANUP ==============
setInterval(async () => {
  const now = Date.now();
  const toDelete = [];
  
  for (const [clientId, clientData] of activeClients.entries()) {
    if (now - clientData.lastSeen > 10 * 60 * 1000 && !clientData.offline) {
      const offlineEmbed = new EmbedBuilder()
        .setTitle("I2 Data Connection Ended")
        .setDescription(`An i2 for \`${clientData.location.cityNm}, ${clientData.location.stCd}\` has disconnected from STARSRV.\nThis connection was last seen at <t:${Math.round(clientData.lastSeen / 1000)}>.\nConnection ID: \`${clientData.id}\``)
        .setFooter({ text: "RWE v2 - STARSRV US CENTRAL 1" })
        .setTimestamp()
        .setColor("Orange");
      
      webhook.send({ embeds: [offlineEmbed] }).catch(e => console.error('Webhook send failed:', e.message));
      activeClients.set(clientId, { ...clientData, offline: true });
    }
    
    if (now - clientData.lastSeen > 10 * 60 * 1000) {
      toDelete.push({ clientId, filePath: clientData.filePath });
    }
  }
  
  for (const { clientId, filePath } of toDelete) {
    try {
      await fs.promises.unlink(filePath);
      activeClients.delete(clientId);
    } catch (e) {
      if (e.code !== 'ENOENT') console.error(`Failed to delete ${filePath}:`, e.message);
      activeClients.delete(clientId);
    }
  }
  
  await cleanupOldFiles(uploadsDir, 2 * 60 * 60 * 1000);
  await cleanupOldFiles(uploadsDir_i1, 2 * 60 * 60 * 1000);
}, 10 * 60 * 1000);

// ============== PUBLISH HANDLERS ==============
async function handleMachineProductCfg(client, packet, callback, clientSubscriptions) {
  const payload = packet.payload.toString();
  const sizeKB = Buffer.byteLength(payload) / 1024;

  if (sizeKB > 50) {
    console.log("mpc received packet over 50kb");
    return callback(new Error('Payload exceeds 50KB'));
  }

  let parsed;
  try {
    parsed = JSON.parse(payload);
    const parser = new XMLParser();
    parser.parse(parsed.data);
  } catch {
    return callback(new Error('Payload is not valid XML'));
  }

  const clientIdSafe = (client.id || 'client')
    .replace(/[\\/]/g, '-')
    .replace(/_/g, '-');

  const existingFile = [...activeClients.entries()].find(([id]) => id === clientIdSafe);
  if (existingFile) {
    try {
      fs.unlinkSync(existingFile[1].filePath);
    } catch {}
  }

  const filePath = path.join(uploadsDir, `${Date.now()}_${clientIdSafe}.xml`);
  fs.writeFileSync(filePath, parsed.data);
  
  const location = parsed.data.split("<ConfigItem key=\"PrimaryLocation\" value=\"")[1].split("\"")[0];

  const lfrData = await require("../schemas").istar_lfrecord("locId", location)
  const primZone = (parsed.data.split("<ConfigItem key=\"primaryZone\" value=\"")[1]?.split("\"")[0] || lfrData.zoneId);
  const primCnty = (parsed.data.split("<ConfigItem key=\"primaryCounty\" value=\"")[1]?.split("\"")[0] || lfrData.cntyId);

  const id = Math.round(Math.random() * 1000000).toString();
    let starType = "";
    let starId = ``
  
  try {
    const cues = JSON.parse(fs.readFileSync(path.join(__dirname, "cues.json"), "utf-8"));
    const subs = clientSubscriptions.get(client.id) || new Set();

    for (const [cueId, cueObj] of Object.entries(cues)) {
      if (subs.has(cueObj.topic)) {
        starId = ` - \`${cueId}\``
        starType = String(cueObj.unitType).toUpperCase();
        cueObj.location = { display: `${lfrData.cityNm}, ${lfrData.stCd}`, primZone, primCnty };
      }
    }

    fs.writeFileSync(path.join(__dirname, "cues.json"), JSON.stringify(cues, null, 2));
  } catch (e) {
    console.error("Failed to update cues.json:", e);
  }

  if (!activeClients.has(clientIdSafe)) {
    const onlineEmbed = new EmbedBuilder()
      .setTitle("I2 Data Connection Started")
      .setDescription(`An i2${starType} for \`${lfrData.cityNm}, ${lfrData.stCd}\` has connected to STARSRV.\nThis connection was started at <t:${Math.round(new Date() / 1000)}>.\nConnection ID: \`${id}\`${starId}`)
      .setFooter({ text: "RWE v2 - STARSRV US CENTRAL 1" })
      .setTimestamp()
      .setColor("Green");
    
    webhook.send({ embeds: [onlineEmbed] });
  }
  
  activeClients.set(clientIdSafe, { filePath, lastSeen: Date.now(), location: lfrData, offline: false, id });
  updateLocations(parsed.data);
  
  return callback(null);
}

async function handleI1Config(client, packet, callback) {
  const payload = packet.payload.toString();
  const sizeKB = Buffer.byteLength(payload) / 1024;

  if (sizeKB > 200) {
    console.log("i1 config received packet over 200kb");
    return callback(new Error('Payload exceeds 200KB'));
  }

  const clientIdSafe = (client.id || 'client')
    .replace(/[\\/]/g, '-')
    .replace(/_/g, '-');

  const existingFile = [...activeClients.entries()].find(([id]) => id === clientIdSafe);
  if (existingFile) {
    try {
      fs.unlinkSync(existingFile[1].filePath);
    } catch {}
  }

  const filePath = path.join(uploadsDir_i1, `${Date.now()}_${clientIdSafe}.py`);
  fs.writeFileSync(filePath, payload);
  
  const location = payload.split("dsm.set('primaryCoopId','")[1].split("'")[0];

  const lfrData = await require("../schemas").istar_lfrecord("coop", location)
  const primZone = (payload.split("dsm.set('primaryZone','")[1]?.split("'")[0] || lfrData.zoneId);
  const primCnty = (payload.split("dsm.set('primaryCounty','")[1]?.split("'")[0] || lfrData.cntyId);
  const headendName = (payload.split("dsm.set('headendName','")[1]?.split("'")[0] || lfrData.cntyId);
  const mso = (payload.split("dsm.set('msoName','")[1]?.split("'")[0] || lfrData.cntyId);

  const id = Math.round(Math.random() * 1000000).toString();

  if (!activeClients.has(clientIdSafe)) {
    const onlineEmbed = new EmbedBuilder()
      .setTitle("I1 Data Connection Started")
      .setDescription(`An i1 for \`${lfrData.cityNm}, ${lfrData.stCd}\` has connected to STARSRV.\nThis connection was started at <t:${Math.round(new Date() / 1000)}>.\nConnection ID: \`${id}\`\nUnit: \`${headendName}\` - \`${mso}\``)
      .setFooter({ text: "RWE v2 - STARSRV US CENTRAL 1" })
      .setTimestamp()
      .setColor("Green");
    
    webhook.send({ embeds: [onlineEmbed] });
  }
  
  activeClients.set(clientIdSafe, { filePath, lastSeen: Date.now(), location: lfrData, offline: false, id });
  updateLocations_i1(payload);
  
  return callback(null);
}

// ============== LOCATION UPDATERS ==============
function updateLocations(xmlStr) {
  const parser = new XMLParser({
    ignoreAttributes: false,
    attributeNamePrefix: '',
  });

  let xml;
  try {
    xml = parser.parse(xmlStr);
  } catch (e) {
    console.error('Failed to parse XML:', e);
    return;
  }

  const keysToCheck = [
    'LocalRadarCity', 'MapCity', 'MetroMapCity', 'NearbyLocation',
    'RegionalMapCity', 'WinterGetawayLocation', 'TravelCity', 'SummerGetawayLocation'
  ];

  const keysToCheckForTides = ['TideStation'];

  const currentGeneral = new Set();
  const currentAlerts = new Set();
  const currentTides = new Set();

  let configItems = [];
  try {
    const items = xml?.Config?.ConfigDef?.ConfigItems?.ConfigItem;
    if (!items) return;
    configItems = Array.isArray(items) ? items : [items];
  } catch {
    return;
  }

  for (const item of configItems) {
    const key = item.key || item['key'] || item['@_key'] || null;
    const value = item.value || item['value'] || item['@_value'] || null;
    if (!key || !value) continue;

    for (let i = 1; i <= 8; i++) {
      for (const baseKey of keysToCheck) {
        if (key === `${baseKey}${i}`) currentGeneral.add(value);
      }
      for (const baseKey of keysToCheckForTides) {
        if (key === `${baseKey}${i}`) currentTides.add(value);
      }
    }
    
    if (key === 'PrimaryLocation') currentGeneral.add(value);
    if (key === 'primaryZone') currentAlerts.add(value);
    if (key === 'primaryCounty') currentAlerts.add(value);
    if (key === 'secondaryZones') {
      value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
    }
    if (key === 'secondaryCounties') {
      value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
    }
  }

  for (const clientData of activeClients.values()) {
    try {
      const fileData = fs.readFileSync(clientData.filePath, 'utf-8');
      const parsedXml = parser.parse(fileData);
      const items = parsedXml?.Config?.ConfigDef?.ConfigItems?.ConfigItem;
      const allItems = Array.isArray(items) ? items : [items];

      for (const item of allItems) {
        const key = item.key || item['key'] || item['@_key'] || null;
        const value = item.value || item['value'] || item['@_value'] || null;
        if (!key || !value) continue;

        for (let i = 1; i <= 8; i++) {
          for (const baseKey of keysToCheck) {
            if (key === `${baseKey}${i}`) currentGeneral.add(value);
          }
          for (const baseKey of keysToCheckForTides) {
            if (key === `${baseKey}${i}`) currentTides.add(value);
          }
        }
        
        if (key === 'PrimaryLocation') currentGeneral.add(value);
        if (key === 'primaryZone') currentAlerts.add(value);
        if (key === 'primaryCounty') currentAlerts.add(value);
        if (key === 'secondaryZones') {
          value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
        }
        if (key === 'secondaryCounties') {
          value.split(',').map(v => v.trim()).forEach(v => currentAlerts.add(v));
        }
      }
    } catch (e) {
      console.warn("Failed to parse active client's XML:", e.message);
    }
  }

  let locations = { general: [], alerts: [], tides: [], national: [] };
  try {
    locations = JSON.parse(fs.readFileSync(locationsFile, 'utf8'));
  } catch {}

  locations.general = [...currentGeneral].sort();
  locations.alerts = [...currentAlerts].sort();
  locations.tides = [...currentTides].sort();

  fs.writeFileSync(locationsFile, JSON.stringify(locations, null, 2));
}

function updateLocations_i1(pythonConfig) {
  const currentGeneral = JSON.parse(
    pythonConfig
      .split("wxdata.setInterestList('obsStation','1',")[1]
      .split(")")[0]
      .replace(",]", "]")
      .replaceAll("'", '"')
  );

  const currentCoop = JSON.parse(
    pythonConfig
      .split("wxdata.setInterestList('coopId','1',")[1]
      .split(")")[0]
      .replace(",]", "]")
      .replaceAll("'", '"')
  );

  const currentSki = JSON.parse(
    pythonConfig
      .split("wxdata.setInterestList('skiId','1',")[1]
      .split(")")[0]
      .replace(",]", "]")
      .replaceAll("'", '"')
  );

  let locations = { i1: { obs: [], coop: [], ski: [] }, general: [], alerts: [], tides: [], national: [] };

  try {
    locations = JSON.parse(fs.readFileSync(locationsFile, 'utf8'));
    locations.i1 = locations.i1 || { obs: [], coop: [], ski: [] };
  } catch {}

  locations.i1.obs = Array.from(new Set([...(locations.i1.obs || []), ...currentGeneral])).sort();
  locations.i1.coop = Array.from(new Set([...(locations.i1.coop || []), ...currentCoop])).sort();
  locations.i1.ski = Array.from(new Set([...(locations.i1.ski || []), ...currentSki])).sort();

  fs.writeFileSync(locationsFile, JSON.stringify(locations, null, 2));
}

const auth = require("../auth/base/encoder-auth")

// ============== EXPORTS ==============
module.exports = {
  async handlePublish(client, packet, callback, clientSubscriptions) {
    const topic = packet.topic;
    
    if (topic === 'connection/machineproductcfg') {
      return handleMachineProductCfg(client, packet, callback, clientSubscriptions);
    }
    
    if (topic === 'connection/i1config') {
      return //handleI1Config(client, packet, callback);
    }
    
    return callback(new Error('Not authorized to publish to this topic'));
  },

  handleClientDisconnect(client) {
    activeClients.delete(client.id);
  },

  async authorize(client, username, password, callback) {
    const secret = username?.toString();
    if (!secret) return callback(new Error('No secret provided'), false);

    try {
        const authenticated = await auth(secret)
        if(authenticated == true) {
          client.secret = secret;
          return callback(null, true);
        } else {
          return callback(new Error('You are not authorized.'), false)
        }
    } catch (err) {
        console.error('Auth error:', err.message);
        callback(new Error('Auth server error'), false);
    }
    }
};