const fs = require("fs");
const path = require("path");
const schemaSys = require("../../../schemas");
const { sendI2Data, disconnect } = require("../../mqttlib");

const CACHE_FILE = path.join(__dirname, "dataCache.json");

// Load cache
let cache = {};
if (fs.existsSync(CACHE_FILE)) {
  try {
    cache = JSON.parse(fs.readFileSync(CACHE_FILE, "utf-8"));
  } catch (err) {
    console.error("⚠️ Error reading cache:", err);
  }
}

const dataTypes = [
  {
    name: "Aches and Pains",
    dataType: "AchesAndPains",
    module: schemaSys.i2_data_aches,
    combine: schemaSys.i2_res_aches,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
  {
    name: "Airport Delays",
    dataType: "AirportDelays",
    module: schemaSys.i2_data_airport,
    combine: schemaSys.i2_res_airport,
    locationChangeRegen: false,
    storeTime: 15 * 60,
    priority: 5
  },
  {
    name: "Air Quality",
    dataType: "AirQuality",
    module: schemaSys.i2_data_aqi,
    combine: schemaSys.i2_res_aqi,
    locationChangeRegen: true,
    storeTime: 30 * 60,
    priority: 1
  },
  {
    name: "I2 Alerts",
    dataType: "BERecord",
    module: schemaSys.i2_data_berecord,
    combine: schemaSys.i2_res_berecord,
    locationChangeRegen: false,
    storeTime: 2 * 60,
    priority: 10
  },
  {
    name: "Breathing",
    dataType: "Breathing",
    module: schemaSys.i2_data_breathing,
    combine: schemaSys.i2_res_breathing,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
  {
    name: "Current Observations",
    dataType: "CurrentObservations",
    module: schemaSys.i2_data_cc,
    combine: schemaSys.i2_res_cc,
    locationChangeRegen: true,
    storeTime: 15 * 60,
    priority: 5
  },
  {
    name: "Daily Forecast",
    dataType: "DailyForecast",
    module: schemaSys.i2_data_dailyfcst,
    combine: schemaSys.i2_res_dailyfcst,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 3
  },
  {
    name: "Dry Skin Index",
    dataType: "DrySkin",
    module: schemaSys.i2_data_dryskin,
    combine: schemaSys.i2_res_dryskin,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
  {
    name: "Heating and Cooling",
    dataType: "HeatingAndCooling",
    module: schemaSys.i2_data_heatingcooling,
    combine: schemaSys.i2_res_heatingcooling,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
  {
    name: "Hourly Forecast",
    dataType: "HourlyForecast",
    module: schemaSys.i2_data_hourlyfcst,
    combine: schemaSys.i2_res_hourlyfcst,
    locationChangeRegen: true,
    storeTime: 30 * 60,
    priority: 3
  },
  {
    name: "Mosquito Index",
    dataType: "Mosquito",
    module: schemaSys.i2_data_mosquito,
    combine: schemaSys.i2_res_mosquito,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
  {
    name: "Pollen Forecast",
    dataType: "PollenForecast",
    module: schemaSys.i2_data_pollenfcst,
    combine: schemaSys.i2_res_pollenfcst,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  },
/*   {
    name: "Pollen Observations",
    dataType: "PollenObs",
    module: schemaSys.i2_data_pollenobs,
    combine: schemaSys.i2_res_pollenobs,
    locationChangeRegen: true,
    storeTime: 15 * 60,
    priority: 2
  }, */
  {
    name: "Watering Needs",
    dataType: "WateringNeeds",
    module: schemaSys.i2_data_wateringneeds,
    combine: schemaSys.i2_res_wateringneeds,
    locationChangeRegen: true,
    storeTime: 60 * 60,
    priority: 1
  }
];

// Helpers
const isExpired = (timestamp, ttl) => Date.now() - timestamp > ttl * 1000;
const hasLocationChanged = (prevHash, currentLocations) => prevHash !== JSON.stringify(currentLocations);

async function processModule(item) {
  const key = item.dataType;
  const cacheEntry = cache[key] || {};
  const locations = JSON.parse(fs.readFileSync(path.join(__dirname, "../../locations.json"), 'utf-8'))
  const locationsHash = JSON.stringify({ general: locations.general, national: locations.national });

  const shouldRegen =
    !cacheEntry.timestamp ||
    isExpired(cacheEntry.timestamp, item.storeTime) ||
    (item.locationChangeRegen && hasLocationChanged(cacheEntry.locationHash, locationsHash));

  if (!shouldRegen) {
    console.log(`Using cached data for ${key}`);
    await sendI2Data(item.dataType, cacheEntry.data, "i2/data");
    return;
  }

  console.log(`Running module: ${key}`);

  try {
    let allResults = [];

    if (item.locationChangeRegen) {
      // Run per-location
      for (const loc of locations.general) {
        try {
          const result = await item.module(loc, loc);
          if (result) allResults.push(result);
        } catch (err) {
          console.warn(`Error in ${key} (general ${loc}):`, err.message);
        }
      }

      for (const loc of locations.national) {
        try {
          const result = await item.module(loc, loc);
          if (result) allResults.push(result);
        } catch (err) {
          console.warn(`Error in ${key} (national ${loc}):`, err.message);
        }
      }
    } else {
      // Run once for all locations
      try {
        const result = await item.module(locations.general, locations.national);
        if (result) allResults.push(result);
      } catch (err) {
        console.warn(`Error running ${key} (all locations):`, err.message);
      }
    }

    // Combine if applicable
    let finalData = allResults;
    if (item.combine) {
      try {
        finalData = await item.combine(allResults);
      } catch (err) {
        console.warn(`Combine failed for ${key}:`, err.message);
      }
    }

    // Cache
    cache[key] = {
      timestamp: Date.now(),
      locationHash: locationsHash,
      data: finalData
    };
    fs.writeFileSync(CACHE_FILE, JSON.stringify(cache, null, 2));

    console.log(`Sending ${key} (${item.dataType}) via MQTT...`);
    await sendI2Data(item.dataType, finalData, "i2/data");

    console.log(`Cached + sent ${key}`);
  } catch (err) {
    console.error(`Error running ${key}:`, err);
  }
}

async function runAll() {
  console.log(`\nRunning data collection at ${new Date().toLocaleString()}`);
  for (const item of dataTypes.sort((a, b) => b.priority - a.priority)) {
    await processModule(item);
  }
  console.log("Cycle complete.");
  disconnect();
}

module.exports = runAll
