const lib = require("../../mqttlib");
const schemas = require("../../../schemas");
const path = require("path");

const radarTypes = {
  "radar.US": { type: "Radar", location: "US" },
  "radar.AK": { type: "Radar", location: "AK" },
  "radar.HI": { type: "Radar", location: "HI" },
  "radar.PR": { type: "Radar", location: "PR" },
  "satrad.US": { type: "SatRad", location: "US" }
};

let running = false;
let connected = false;
let disconnectTimer = null;

async function ensureConnected() {
  if (!connected) {
    await lib.connect();
    connected = true;
    console.log("[Radar] Connected to lib");
  }

  clearTimeout(disconnectTimer);
  disconnectTimer = setTimeout(async () => {
    await lib.disconnect();
    connected = false;
    console.log("[Radar] Disconnected after idle timeout");
  }, 30 * 1000);
}

async function getRadarAndSend() {
  if (running) {
    console.log("[Radar] Skipping run — previous still running");
    return;
  }

  running = true;

  try {
    const mapDataCache = schemas.images_maps_data();
    const timestampsReqCache = schemas.images_maps_content();
    await ensureConnected();

    for (const type in radarTypes) {
      const radarType = radarTypes[type];
      let mapData = null;

      for (const key in mapDataCache) {
        if (mapDataCache[key]?.starName === type) {
          mapData = mapDataCache[key];
          break;
        }
      }

      if (!mapData) {
        console.warn(`[Radar] No mapData found for ${type}`);
        continue;
      }

      const timestamps = timestampsReqCache[mapData.apiName] || [];
      if (timestamps.length === 0) {
        console.warn(`[Radar] No timestamps for ${mapData.apiName}`);
        continue;
      }

      for (const ts of timestamps) {
        const location = type.split(".")[1];
        const filename = path.basename(ts);

        try {
          await lib.sendI1Radar(type.split(".")[1].replace("satrad", "radarSatellite"), ts, ts.split(".tiff")[0].split("/").reverse()[0].replace("satrad", "radarSatellite") + ".tif", radarType.type.replace("satrad", "radarSatellite"),  ts.split(".tiff")[0].split("/").reverse()[0].replace("satrad", "radarSatellite") + ".tif")
          console.log(`[Radar] Sent ${filename} for ${type}`);
        } catch (err) {
          console.error(`[Radar] Error sending ${filename} for ${type}:`, err);
        }
      }
    }

    clearTimeout(disconnectTimer);
    disconnectTimer = setTimeout(async () => {
      await lib.disconnect();
      connected = false;
      console.log("[Radar] Disconnected after idle timeout");
    }, 30 * 1000);
  } catch (err) {
    console.error("[Radar] getRadarAndSend() error:", err);
  } finally {
    running = false;
  }
}

module.exports = getRadarAndSend;
