import base64
import os
import json
import paho.mqtt.client as mqtt
from datetime import datetime, timezone
import xml.etree.ElementTree as ET
import subprocess
import shutil
import sys
import threading
import traceback
import time
import socket
import struct
import ctypes
import requests
import queue
from pathlib import Path

# Thread-safe globals using threading.Lock
config_lock = threading.Lock()
API_SERVER = ""
apiKey = ""
APP_VERSION = "1.0.6"
CONFIG_FILE = "config.xml"
IMAGE_TOPIC = "i1/radar"
CACHE_FILE = "cache"
cached_filenames = set()
cache_lock = threading.Lock()

cfg_path = "config.py"
mqtt_topic = "connection/i1config"
cache_timestamp = None
isConfigLocal = True
configs_dir = "configs"

# Thread-safe UDP socket with lock
udp_socket = None
udp_lock = threading.Lock()

# Thread-safe message ID counter
msgid_lock = threading.Lock()
current_msgid = 1

# Thread-safe MQTT publish queue
mqtt_publish_queue = queue.Queue(maxsize=1000)

LOG_LEVELS = {"DEBUG": 0, "LOG": 1}
current_log_level = LOG_LEVELS["DEBUG"]
log_lock = threading.Lock()

def log(msg, level="DEBUG"):
    """Thread-safe logging with lock"""
    level = level.upper()
    if level not in LOG_LEVELS:
        level = "DEBUG"
    if LOG_LEVELS[level] >= current_log_level:
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with log_lock:
            print(f"[{timestamp}] {msg}", flush=True)

def safe_thread_wrapper(func, *args, **kwargs):
    """Wrapper to catch all exceptions in threads"""
    try:
        func(*args, **kwargs)
    except Exception as e:
        log(f"[-] THREAD CRASH in {func.__name__}: {e}", level="LOG")
        log(f"[-] Traceback: {traceback.format_exc()}", level="DEBUG")

def check_for_updates(api_url, current_version, api_key):
    try:
        if not api_url or not api_key:
            return False
        url = f"{api_url}/api/version?apiKey={api_key}"
        r = requests.get(url, timeout=15)
        if r.status_code != 200:
            log(f"[-] Version check failed: {r.status_code}", level="DEBUG")
            return False

        version_data = r.json()
        latest_version = version_data.get("srv", "")
        if latest_version and latest_version != current_version:
            log(f"[UPDATE] New version available: {latest_version}", level="LOG")
            return True
    except Exception as e:
        log(f"[-] Error checking for update: {e}", level="DEBUG")
    return False

def read_cfg_file(path):
    """Reads a file and returns its contents, or an error message wrapped in HTML comment."""
    try:
        with open(path, "r", encoding="utf-8") as f:
            return f.read()
    except Exception as e:
        return f"<!-- Error reading file: {e} -->"

def get_config():
    ensure_temp_dir()
    transport = paramiko.Transport((ssh_config["hostname"], ssh_config["port"]))
    transport.connect(username=ssh_config["username"], password=ssh_config["password"])
    sftp = paramiko.SFTPClient.from_transport(transport)

    local_path = "config.py"
    remote_path = "/usr/home/dgadmin/config/current/config.py"
    sftp.get(remote_path, local_path)
    #print("i1DT - Config downloaded from i1.")
    sftp.close()
    transport.close()

def mqtt_publish_worker(client):
    """Dedicated thread for MQTT publishing - ensures thread safety"""
    log("[WORKER] MQTT publish worker started", level="DEBUG")
    while True:
        try:
            topic, payload = mqtt_publish_queue.get(timeout=1)
            if topic is None:  # Poison pill to stop thread
                log("[WORKER] MQTT publish worker stopping", level="DEBUG")
                break
            result = client.publish(topic, payload)
            if result.rc != mqtt.MQTT_ERR_SUCCESS:
                log(f"[-] MQTT publish failed: rc={result.rc} topic={topic}", level="DEBUG")
            else:
                log(f"[PUB] Published to {topic}: {len(payload)} bytes", level="DEBUG")
            mqtt_publish_queue.task_done()
        except queue.Empty:
            continue
        except Exception as e:
            log(f"[-] Error in MQTT publish worker: {e}", level="DEBUG")
            log(f"[-] Traceback: {traceback.format_exc()}", level="DEBUG")

def safe_mqtt_publish(topic, payload):
    """Thread-safe MQTT publish using queue"""
    try:
        mqtt_publish_queue.put((topic, payload), timeout=1)
    except queue.Full:
        log(f"[-] MQTT publish queue full, dropping message", level="DEBUG")
    except Exception as e:
        log(f"[-] Error queuing MQTT publish: {e}", level="DEBUG")

def send_cfg(client, apiServer, key):
    get_config()
    global cache_timestamp

    configs_path = Path(configs_dir)
    cfg_file_path = Path(cfg_path)

    # Check for XML files in configs_dir
    if configs_path.exists() and configs_path.is_dir():
        xml_files = [f for f in configs_path.glob("*.xml") if f.is_file()]

        if xml_files:
            xml_files.sort(key=lambda f: f.name)
            for f in xml_files[:5]:  # Limit to 5 files
                file_data = read_cfg_file(f)
                payload = {
                    "timestamp": int(time.time()),
                    "filename": f.name,
                    "data": file_data
                }
                client.publish(mqtt_topic, json.dumps(payload))
            # Update cache timestamp to now after sending all files
            cache_timestamp = int(time.time())
            return

    # Fallback: send local config file if it exists
    if cfg_file_path.exists():
        mod_time = cfg_file_path.stat().st_mtime
        if mod_time != cache_timestamp:
            data = read_cfg_file(cfg_file_path)
            payload = {
                "timestamp": int(time.time()),
                "filename": cfg_file_path.name,
                "data": data
            }
            client.publish(mqtt_topic, json.dumps(payload))
            cache_timestamp = mod_time

def update_check_loop(apiServer, key):
    """Thread-safe update check loop"""
    while True:
        try:
            if check_for_updates(apiServer, APP_VERSION, key):
                log("[UPDATE] Downloading new version...", level="LOG")
                download_and_launch_updater(apiServer, key)
                sys.exit(0)
            time.sleep(300)
        except Exception as e:
            log(f"[-] Error in update_check_loop: {e}", level="DEBUG")
            time.sleep(300)

def monitor_cfg_file(client, apiServer, key):
    """Thread-safe config monitor"""
    while True:
        try:
            time.sleep(2)
            send_cfg(client, apiServer, key)
            time.sleep(8)
        except Exception as e:
            log(f"[-] Error in monitor_cfg_file: {e}", level="DEBUG")
            time.sleep(10)

def keepalive_cfg(client):
    while True:
        data = ""

        if isConfigLocal:
            get_config()
            if os.path.exists(cfg_path):
                data = read_cfg_file(cfg_path)
        else:
            os.makedirs(configs_dir, exist_ok=True)
            files = [f for f in Path(configs_dir).glob("*.xml") if f.is_file()]
            files.sort(key=lambda f: f.name)
            files = files[:5]

            for f in files:
                file_data = read_cfg_file(f)
                data += f"\n<!-- {f.name} -->\n{file_data}\n"

        if data.strip():
            payload = {
                "timestamp": int(time.time()),
                "data": data.strip()
            }
            try:
                client.publish(mqtt_topic, json.dumps(payload))
            except Exception as e:
                print(e)
                log(f"The encoder pmo", level="LOG")

        time.sleep(300)  # wait 5 minutes

def load_cache():
    global cached_filenames
    try:
        with cache_lock:
            if os.path.exists(CACHE_FILE):
                with open(CACHE_FILE, "r", encoding="utf-8") as f:
                    cached_filenames = set(line.strip() for line in f if line.strip())
    except Exception as e:
        log(f"[-] Error loading cache: {e}", level="DEBUG")
        cached_filenames = set()

def add_to_cache(filename):
    global cached_filenames
    try:
        with cache_lock:
            if filename not in cached_filenames:
                cached_filenames.add(filename)
                with open(CACHE_FILE, "a", encoding="utf-8") as f:
                    f.write(filename + "\n")
                log(f"[CACHE] Added {filename} to cache", level="DEBUG")
    except Exception as e:
        log(f"[-] Failed to write to cache file: {e}", level="DEBUG")

def is_in_cache(filename):
    try:
        with cache_lock:
            return filename in cached_filenames
    except Exception as e:
        log(f"[-] Error checking cache: {e}", level="DEBUG")
        return False

def get_next_msgid():
    """Thread-safe message ID generation"""
    global current_msgid
    with msgid_lock:
        try:
            # Try to load from file
            if os.path.exists('./msgId.txt'):
                with open('./msgId.txt', 'r') as f:
                    content = f.read().strip()
                    if content:
                        current_msgid = int(content)
        except Exception as e:
            log(f"[-] Error reading msgId.txt: {e}", level="DEBUG")

        msgid = current_msgid
        current_msgid += 1

        # Save new ID
        try:
            with open('./msgId.txt', 'w') as f:
                f.write(str(current_msgid))
        except Exception as e:
            log(f"[-] Error writing msgId.txt: {e}", level="DEBUG")

        return msgid

def create_default_config():
    default_xml = """<?xml version="1.0" encoding="UTF-8"?>
<mqtt>
    <server>mqtt.example.com</server>
    <port>1883</port>
    <apiKey>YOUR_API_KEY_HERE</apiKey>
    <topics>
        <topic>i1/radar</topic>
        <topic>i1/data</topic>
    </topics>
    <tls>False</tls>
    <ipAddr>127.0.0.1</ipAddr>
    <logLevel>DEBUG</logLevel>
</mqtt>
"""
    with open(CONFIG_FILE, "w", encoding="utf-8") as f:
        f.write(default_xml)
    print(f"[CONFIG] {CONFIG_FILE} created with default values. Please edit it and rerun.")
    sys.exit(0)

def parse_bool(value, default=True):
    if value is None:
        return default
    return str(value).strip().lower() == "true"

def load_config():
    if not os.path.exists(CONFIG_FILE):
        create_default_config()

    tree = ET.parse(CONFIG_FILE)
    root = tree.getroot()

    if root.tag != "mqtt":
        print(f"[-] Invalid {CONFIG_FILE}: root element is not <mqtt>")
        sys.exit(1)

    server = root.findtext("server", default="mqtt.example.com")
    port = int(root.findtext("port", default="1883"))
    tls = parse_bool(root.findtext("tls"), default=True)
    apiKey = root.findtext("apiKey", default="")
    udp = parse_bool(root.findtext("enableUDP"), default=False)
    mcastIf = root.findtext("ipAddr") or "127.0.0.1"

    topics_node = root.find("topics")
    topics = [t.text for t in topics_node.findall("topic")] if topics_node is not None else []

    config = {
        "server": server,
        "port": port,
        "apiKey": apiKey,
        "topics": topics,
        "tls": tls,
        "udp": udp,
        "udpAddr": mcastIf,
        "MCAST_GRP": mcastIf,
        "MCAST_IF": mcastIf,
        "BUF_SIZE": 1396,
        "MULTICAST_TTL": 255
    }

    return config

def init_udp_socket(config):
    """Initialize UDP socket with error handling and thread safety"""
    global udp_socket
    try:
        if not config["udp"]:
            log("[-] UDP Multicast is disabled in the config.", level="DEBUG")
            return None

        with udp_lock:
            MCAST_GRP = config["MCAST_GRP"]
            MCAST_IF = config["MCAST_IF"]
            MULTICAST_TTL = config.get("MULTICAST_TTL", 2)

            udp_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
            udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            udp_socket.setsockopt(socket.SOL_IP, socket.IP_MULTICAST_TTL, MULTICAST_TTL)
            udp_socket.setsockopt(socket.SOL_IP, socket.IP_MULTICAST_IF, socket.inet_aton(MCAST_IF))
            udp_socket.setsockopt(socket.SOL_IP, socket.IP_ADD_MEMBERSHIP,
                           socket.inet_aton(MCAST_GRP) + socket.inet_aton(MCAST_IF))

            log("[+] Multicast socket setup complete.", level="LOG")
            log(f"MCAST_IF: {MCAST_IF} | MCAST_GRP: {MCAST_GRP}", level="DEBUG")
            return udp_socket
    except Exception as e:
        log(f"[-] Failed to initialize UDP socket: {e}", level="LOG")
        return None

def safe_udp_send(data, address):
    """Thread-safe UDP send with lock"""
    global udp_socket
    try:
        with udp_lock:
            if udp_socket is None:
                log(f"[-] UDP socket not initialized", level="DEBUG")
                return False
            udp_socket.sendto(data, address)
            return True
    except Exception as e:
        log(f"[-] UDP send error: {e}", level="DEBUG")
        return False
import paramiko
with open("config.json", "r") as f:
    ssh_config = json.load(f).get("ssh", {})
ssh_connected = False
ssh_client = None
shell = None
def connect_ssh():
    global ssh_client, shell, ssh_connected
    ssh_client = paramiko.SSHClient()
    ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
    ssh_client.connect(
        ssh_config["hostname"],
        port=ssh_config["port"],
        username=ssh_config["username"],
        password=ssh_config["password"],
        look_for_keys=False,
        allow_agent=False
    )
    ssh_connected = True
    shell = ssh_client.invoke_shell()
    log("RWE - SSH connection established.", level="DEBUG")

    def handle_output():
        while True:
            password = ssh_config["password"]
            output = shell.recv(1024).decode()
            if "Password:" in output:
                shell.send(f"{password}\n")

    threading.Thread(target=handle_output, daemon=True).start()
    shell.send("su -l dgadmin\n")

def send_command(command):
    if not ssh_connected:
        connect_ssh()
    #print(f"RWE - Sent command: {command}")
    shell.send(command + "\n")

def sync_time():
    """Sync time with the i1 system every hour"""
    try:
        now = datetime.now()
        freebsd_timestamp = now.strftime("%m%d%H%M%Y.%S")
        log(f"[SYNC] Syncing time, timestamp: {freebsd_timestamp}", level="LOG")
        send_command(f"date {freebsd_timestamp}")
    except Exception as e:
        log(f"[-] Error syncing time: {e}", level="DEBUG")

def time_sync_loop():
    """Thread-safe hourly time sync loop"""
    while True:
        try:
            time.sleep(3600)  # Wait 1 hour
            sync_time()
        except Exception as e:
            log(f"[-] Error in time_sync_loop: {e}", level="DEBUG")
            time.sleep(3600)

def ensure_temp_dir():
    #print("ok")
    if not os.path.exists("temp"):
        os.makedirs("temp")
work_queue = queue.Queue()
max_retries = 10

def upload_file(file):
    """Handles uploading a single file via SFTP."""
    ensure_temp_dir()
    transport = None
    try:
        transport = paramiko.Transport((ssh_config["hostname"], ssh_config["port"]))
        transport.connect(username=ssh_config["username"], password=ssh_config["password"])
        sftp = paramiko.SFTPClient.from_transport(transport)

        local_path = os.path.join("temp", file)
        remote_path = f"/home/dgadmin/{file}"
        sftp.put(local_path, remote_path)
        log(f"[RWE] Uploaded {file}", level="DEBUG")

        sftp.close()
        transport.close()
        return True
    except Exception as e:
        log(f"[-] Upload failed for {file}: {e}", level="DEBUG")
        if transport:
            transport.close()
        return False

def process_work_item(item):
    """Process a single queue item with retries."""
    task_type = item["type"]
    data = item["data"]

    for attempt in range(1, max_retries + 1):
        try:
            if task_type == "upload":
                success = upload_file(data)
                if success:
                    log(f"[QUEUE] Upload succeeded for {data} (Attempt {attempt})", level="DEBUG")
                    return
                else:
                    raise Exception("Upload failed")
            elif task_type == "command":
                send_command(data)
                log(f"[QUEUE] Command executed: {data} (Attempt {attempt})", level="DEBUG")
                return
        except Exception as e:
            log(f"[QUEUE] Error ({task_type}) on attempt {attempt}: {e}", level="DEBUG")
            time.sleep(1)
    log(f"[-] {task_type.upper()} failed after {max_retries} attempts: {data}", level="DEBUG")

def worker():
    """Thread that continuously processes the work queue."""
    while True:
        item = work_queue.get()
        if item is None:
            break
        process_work_item(item)
        work_queue.task_done()

def handle_work_request(command):
    """Add a command to the queue."""
    work_queue.put({"type": "command", "data": command})
    log(f"[QUEUE] Added command: {command}", level="DEBUG")

def queue_upload(file):
    """Add an upload task to the queue."""
    work_queue.put({"type": "upload", "data": file})
    log(f"[QUEUE] Added upload: {file}", level="DEBUG")

# === MESSAGE QUEUE FOR ASYNC MQTT PROCESSING ===
message_queue = queue.Queue(maxsize=5000)

def mqtt_message_worker():
    """Dedicated thread for processing MQTT messages asynchronously"""
    log("[WORKER] MQTT message worker started", level="DEBUG")
    while True:
        try:
            client, userdata, msg = message_queue.get(timeout=1)
            if client is None:  # Poison pill to stop thread
                log("[WORKER] MQTT message worker stopping", level="DEBUG")
                break
            handle_message_non_udp(client, userdata, msg)
            message_queue.task_done()
        except queue.Empty:
            continue
        except Exception as e:
            log(f"[-] Error in MQTT message worker: {e}", level="DEBUG")
            log(f"[-] Traceback: {traceback.format_exc()}", level="DEBUG")

# === START WORKER THREADS ===
worker_thread = threading.Thread(target=worker, daemon=True)
worker_thread.start()

mqtt_msg_worker_thread = threading.Thread(target=mqtt_message_worker, daemon=True)
mqtt_msg_worker_thread.start()

def on_connect(client, userdata, flags, reasonCode, properties=None):
    try:
        log(f"[+] Connected to MQTT broker with code {reasonCode}", level="LOG")

        # Subscribe to all topics with QoS 1 for reliability
        for topic in userdata["topics"]:
            result, mid = client.subscribe(topic, qos=1)
            if result == mqtt.MQTT_ERR_SUCCESS:
                log(f"[SUB] Subscribed to topic: {topic} (mid={mid})", level="LOG")
            else:
                log(f"[-] Failed to subscribe to {topic}: rc={result}", level="LOG")

        with config_lock:
            api_server = API_SERVER
            api_key = apiKey

        # Connect to SSH and sync time after all subscriptions complete
        log("[+] All topics subscribed, connecting to SSH...", level="LOG")
        try:
            connect_ssh()
            time.sleep(1.0)  # Give SSH shell time to initialize
            sync_time()
            log("[+] Time synchronized on startup", level="LOG")
        except Exception as e:
            log(f"[-] SSH connection or time sync failed: {e}", level="LOG")

        # Give subscriptions time to process
        time.sleep(0.5)

        send_cfg(client, api_server, api_key)

        # Start other threads with wrapper (but NOT the publish worker - it's already running)
        threading.Thread(target=safe_thread_wrapper,
                        args=(monitor_cfg_file, client, api_server, api_key),
                        daemon=True).start()
        threading.Thread(target=safe_thread_wrapper,
                        args=(keepalive_cfg, client),
                        daemon=True).start()
        threading.Thread(target=safe_thread_wrapper,
                        args=(hourly_resubscribe_loop, client, userdata["topics"]),
                        daemon=True).start()
    except Exception as e:
        log(f"[-] Error in on_connect: {e}", level="LOG")
        log(f"[-] Traceback: {traceback.format_exc()}", level="DEBUG")

def on_disconnect(client, userdata, rc, ok=None, okagain=None):
    log(f"[-] Disconnected from MQTT broker with code {rc}", level="LOG")

    if rc != 0:
        while True:
            try:
                log("[RECONNECT] Attempting to reconnect...", level="LOG")
                time.sleep(5)  # Wait before reconnecting
                client.reconnect()

                if userdata and "topics" in userdata:
                    for topic in userdata["topics"]:
                        client.subscribe(topic)
                        log(f"[RESUB] Re-subscribed to topic: {topic}", level="DEBUG")
                break
            except Exception as e:
                log(f"[-] Reconnect failed: {e}", level="LOG")
                time.sleep(5)

def on_message(client, userdata, msg, properties=None):
    """Thread-safe MQTT message handler - queues messages for async processing"""
    try:
        try:
            message_queue.put((client, userdata, msg), timeout=1)
        except queue.Full:
            log(f"[-] Message queue full, dropping message from {msg.topic}", level="DEBUG")
    except Exception as e:
        log(f"[-] Error in on_message dispatcher: {e}", level="LOG")

def hourly_resubscribe_loop(client, topics):
    """Thread-safe resubscribe loop"""
    while True:
        try:
            now = datetime.now()
            seconds_until_next_hour = 3600 - (now.minute * 60 + now.second)
            time.sleep(seconds_until_next_hour)

            log("[RECONNECT] Re-subscribing to all topics at top of the hour", level="LOG")
            for topic in topics:
                try:
                    client.subscribe(topic)
                    log(f"[RESUB] Re-subscribed to topic: {topic}", level="DEBUG")
                except Exception as e:
                    log(f"[-] Failed to resubscribe to topic {topic}: {e}", level="DEBUG")
        except Exception as e:
            log(f"[-] Error in hourly_resubscribe_loop: {e}", level="DEBUG")
            time.sleep(3600)

def sendMessage(files, commands, numSgmts, Pri, config):
    """Thread-safe UDP message sender"""
    try:
        if Pri == 0:
            MCAST_PORT = 7787
        elif Pri == 1:
            MCAST_PORT = 7788
        else:
            MCAST_PORT = 7787

        MCAST_GRP = config["MCAST_GRP"]
        BUF_SIZE = config["BUF_SIZE"]

        msgNum = get_next_msgid()
        segnmNum = 0

        for filePath, commandOrig in zip(files, commands):
            command = "<MSG><Exec workRequest=\"" + commandOrig + "\" /></MSG>"

            if filePath and os.path.isfile(filePath):
                size = os.path.getsize(filePath)
                import math
                packRounded = math.ceil(size / 1405) + 1
                numSegments = numSgmts + 3

                encode1 = bytes(command + 'I2MSG', 'UTF-8')
                encode2 = len(command).to_bytes(4, byteorder='little')
                commandFooter = encode1 + encode2

                with open(filePath, "ab") as f:
                    f.write(commandFooter)
                new_size = os.path.getsize(filePath)

                p1 = struct.pack(">BHHHIIBBBBBBBIBIBBB", 18, 1, 0, 16, msgNum, 0,
                                segnmNum, 0, 0, 8, numSegments, 3, 0, 0, 8,
                                packRounded, 0, 0, 0)
                safe_udp_send(p1, (MCAST_GRP, MCAST_PORT))

                with open(filePath, "rb") as f:
                    packet_count = 1
                    j = 0
                    while True:
                        data = f.read(BUF_SIZE)
                        if not data:
                            break

                        packetHeader = struct.pack(">BHHHIIBBB", 18, 1, 0, 1405,
                                                   msgNum, packet_count, 0, 0, 0)
                        fec = struct.pack("<IBI", packet_count, 0, new_size)
                        payload = data + bytes(BUF_SIZE - len(data)) if len(
                            data) < BUF_SIZE else data
                        safe_udp_send(packetHeader + fec + payload, (MCAST_GRP, MCAST_PORT))
                        packet_count += 1
                        j += 1

                        if j == 1000:
                            time.sleep(0.1)  # Reduced from 2s
                            j = 0

                segnmNum += 1

            else:
                bx = bytes(command + 'I2MSG', 'utf-8')
                import math
                packRounded = math.ceil(len(bx) / 1405) + 1
                numSegments = 4

                p1 = struct.pack(">BHHHIIBBBBBBBIBIBBB", 18, 1, 0, 16, msgNum, 0,
                                segnmNum, 0, 0, 8, numSegments, 3, 0, 0, 8,
                                packRounded, 0, 0, 0)
                safe_udp_send(p1, (MCAST_GRP, MCAST_PORT))

                packet_count = 1
                j = 0
                new_size = len(bx)

                for offset in range(0, len(bx), BUF_SIZE):
                    chunk = bx[offset:offset + BUF_SIZE]
                    packetHeader = struct.pack(">BHHHIIBBB", 18, 1, 0, 1405,
                                               msgNum, packet_count, 0, 0, 0)
                    fec = struct.pack("<IBI", packet_count, 0, new_size)
                    payload = chunk + bytes(BUF_SIZE - len(chunk)) if len(
                        chunk) < BUF_SIZE else chunk
                    safe_udp_send(packetHeader + fec + payload, (MCAST_GRP, MCAST_PORT))
                    packet_count += 1
                    j += 1
                    if j == 1000:
                        time.sleep(0.1)
                        j = 0

                segnmNum += 1

        for _ in range(3):
            p3 = struct.pack(">BHHHIIBBBBBBBI", 18, 1, 1, 8, msgNum, 0, segnmNum,
                            0, 0, 8, 0, 0, 0, 67108864)
            p4 = struct.pack(">BHHHIIBBB", 18, 1, 1, 14, msgNum, 1, segnmNum, 0,
                            0) + b"hi"
            safe_udp_send(p3, (MCAST_GRP, MCAST_PORT))
            safe_udp_send(p4, (MCAST_GRP, MCAST_PORT))
            segnmNum += 1

        log(f"[SEND] Message {msgNum} sent successfully", level="DEBUG")

    except Exception as e:
        log(f"[-] Error in sendMessage: {e}", level="DEBUG")

def handle_message_non_udp(client, userdata, msg):
    """Thread-safe non-UDP message handler with base64 radar frame decoding"""
    try:
        topic = msg.topic
        payload = msg.payload.decode(errors="ignore")
        log(f"[MSG] Topic: {topic} | {len(payload)} bytes", level="DEBUG")

        try:
            obj = json.loads(payload)
        except json.JSONDecodeError as e:
            log(f"[ERROR] JSON decode failed: {e}", level="DEBUG")
            return

        filename = obj.get("fileName", "")

        # Detect radar/image frames by .tif/.tiff extension
        if filename.endswith(".tif") or filename.endswith(".tiff"):
            try:
                os.makedirs("./temp", exist_ok=True)

                # Normalize to .tif extension
                if filename.endswith(".tiff"):
                    filename = filename.replace(".tiff", ".tif")

                # Check if radar frame already exists to avoid duplicates
                filepath = os.path.join("./temp", filename)
                if os.path.exists(filepath):
                    log(f"[RADAR] Frame already exists, skipping: {filename}", level="DEBUG")
                    return

                b64_data = obj.get("data", "")
                if not b64_data:
                    log("[-] No base64 data found in radar frame payload", level="DEBUG")
                    return

                # Decode base64 to binary
                try:
                    binary_data = base64.b64decode(b64_data)
                except Exception as e:
                    log(f"[-] Failed to decode base64: {e}", level="DEBUG")
                    return

                # Write binary TIFF data
                filepath = os.path.join("./temp", filename)
                with open(filepath, "wb") as f:
                    f.write(binary_data)
                    f.flush()
                    os.fsync(f.fileno())

                queue_upload(filename)
                log(f"[RADAR] Saved binary TIFF to {filepath} ({len(binary_data)} bytes)", level="DEBUG")
                return

            except Exception as e:
                log(f"[-] Failed to save radar frame: {e}", level="DEBUG")
                return

        # Handle generic XML/text data
        try:
            try:
                os.makedirs("./temp", exist_ok=True)
            except Exception as e:
                # idc
                print("idc")

            filename = filename or f"message_{datetime.now().strftime('%Y%m%d_%H%M%S')}.xml"
            xml_data = obj.get("data")

            if not xml_data:
                log("[-] No data found in payload", level="DEBUG")
                handle_work_request(obj.get("workRequest", ""))
                return

            filepath = os.path.join("./temp", filename)
            with open(filepath, "w", encoding="utf-8") as f:
                f.write(xml_data)
                f.flush()
                os.fsync(f.fileno())

            queue_upload(filename)

            workReq = obj.get("workRequest", "").replace("{filepath}", os.path.abspath(filepath))
            handle_work_request(workReq)
            log(f"[XML] Saved data to {filepath}", level="DEBUG")

        except Exception as e:
            print(e)
            log(f"[-] Failed to save text message: {e}", level="DEBUG")

    except Exception as e:
        log(f"[-] Unhandled error in handle_message_non_udp: {e}", level="LOG")
        log(f"[-] Traceback: {traceback.format_exc()}", level="DEBUG")

def download_and_launch_updater(api_server, api_key):
    try:
        update_dir = os.path.join(".", "temp", "update")
        os.makedirs(update_dir, exist_ok=True)

        updater_url = f"{api_server}/api/latest/updater.exe?apiKey={api_key}"
        updater_path = os.path.join(update_dir, "updater.exe")

        encoder_url = f"{api_server}/api/latest/encoder.exe?apiKey={api_key}"
        encoder_path = os.path.join(update_dir, "encoder_update.exe")

        r = requests.get(updater_url, stream=True, timeout=10)
        if r.status_code != 200:
            log(f"[-] Failed to download updater: HTTP {r.status_code}", level="LOG")
            return

        with open(updater_path, "wb") as f:
            for chunk in r.iter_content(1024 * 64):
                f.write(chunk)

        log(f"[UPDATE] Downloaded updater to {updater_path}", level="DEBUG")

        r2 = requests.get(encoder_url, stream=True, timeout=10)
        if r2.status_code != 200:
            log(f"[-] Failed to download encoder update: HTTP {r2.status_code}", level="LOG")
            return

        with open(encoder_path, "wb") as f:
            for chunk in r2.iter_content(1024 * 64):
                f.write(chunk)

        log(f"[UPDATE] Downloaded new encoder to {encoder_path}", level="DEBUG")

        current_exe = sys.executable
        log(f"[UPDATE] Launching updater to replace: {current_exe}", level="LOG")

        subprocess.Popen([updater_path, "-old", current_exe, "-new", encoder_path], shell=True)
        time.sleep(1.0)
        os._exit(0)

    except Exception as e:
        log(f"[-] Update failed: {e}", level="LOG")


def main():
    global API_SERVER, apiKey

    try:
        # Cleanup
        temp_dir = "./temp"
        if os.path.exists(temp_dir):
            try:
                shutil.rmtree(temp_dir)
                log(f"[CLEANUP] Removed existing {temp_dir} directory", level="DEBUG")
            except Exception as e:
                log(f"[-] Failed to remove {temp_dir}: {e}", level="DEBUG")

        os.makedirs(os.path.join(temp_dir, "frames"), exist_ok=True)
        load_cache()

        print(
        r"""
$$$$$$$\  $$\   $$\ $$\      $$\ $$$$$$$$\ $$$$$$$\
$$  __$$\ $$$\  $$ |$$ | $\  $$ |\__$$  __|$$  __$$\
$$ |  $$ |$$$$\ $$ |$$ |$$$\ $$ |   $$ |   $$ |  $$ |
$$$$$$$  |$$ $$\$$ |$$ $$ $$\$$ |   $$ |   $$$$$$$  |
$$  __$$< $$ \$$$$ |$$$$  _$$$$ |   $$ |   $$  __$$<
$$ |  $$ |$$ |\$$$ |$$$  / \$$$ |   $$ |   $$ |  $$ |
$$ |  $$ |$$ | \$$ |$$  /   \$$ |   $$ |   $$ |  $$ |
\__|  \__|\__|  \__|\__/     \__|   \__|   \__|  \__|
$$$$$$$$\                                     $$\
$$  _____|                                    $$ |
$$ |      $$$$$$$\   $$$$$$$\  $$$$$$\   $$$$$$$ | $$$$$$\   $$$$$$\
$$$$$\    $$  __$$\ $$  _____|$$  __$$\ $$  __$$ |$$  __$$\ $$  __$$\
$$  __|   $$ |  $$ |$$ /      $$ /  $$ |$$ /  $$ |$$$$$$$$ |$$ |  \__|
$$ |      $$ |  $$ |$$ |      $$ |  $$ |$$ |  $$ |$$   ____|$$ |
$$$$$$$$\ $$ |  $$ |\$$$$$$$\ \$$$$$$  |\$$$$$$$ |\$$$$$$$\ $$ |
\________|\__|  \__| \_______| \______/  \_______| \_______|\__|


                                                                      """)
        print("Rainwater I2 Data Encoder")
        print("Version 1.0.8 built on 10-11-2025")
        print("Made by Daisy & Susie with love. Thank you for using!")
        print("\n")

        config = load_config()

        # Start auxiliary monitor scripts
        log("[+] Starting auxiliary monitor scripts...", level="LOG")

        # try:
        #     threading.Thread(target=safe_thread_wrapper,
        #                    args=(lambda: subprocess.Popen([sys.executable, "bgm_worker.py"]),),
        #                    daemon=True).start()
        #     log("[+] bgm_worker.py started", level="LOG")
        # except Exception as e:
        #     log(f"[-] Failed to start bgm_worker.py: {e}", level="LOG")

        # Initialize globals safely
        with config_lock:
            API_SERVER = str(config["server"]) + ":" + str(config["port"])
            if config.get("tls") == True:
                API_SERVER = "https://" + API_SERVER
            else:
                API_SERVER = "http://" + API_SERVER
            apiKey = config["apiKey"]

        # Initialize UDP socket if needed
        if config.get("udp"):
            init_udp_socket(config)

        # Check for updates
        if check_for_updates(API_SERVER, APP_VERSION, apiKey):
            log("[UPDATE] Downloading new version...", level="LOG")
            download_and_launch_updater(API_SERVER, apiKey)
            sys.exit(0)

        # Start update check thread
        threading.Thread(target=safe_thread_wrapper,
                        args=(update_check_loop, API_SERVER, apiKey),
                        daemon=True).start()

        # Start hourly time sync thread
        threading.Thread(target=safe_thread_wrapper,
                        args=(time_sync_loop,),
                        daemon=True).start()

        # Setup MQTT
        client = mqtt.Client(mqtt.CallbackAPIVersion.VERSION2,
                           transport="websockets",
                           userdata=config)
        client.username_pw_set(apiKey)

        client.on_connect = on_connect
        client.on_disconnect = on_disconnect
        client.on_message = on_message

        if config.get("tls") == True:
            client.tls_set()

        # Allow slower networks extra time to finish the TLS WebSocket handshake
        client.ws_set_options(path="/mqtt")

        log("[+] Connecting to MQTT broker...", level="LOG")
        client.connect(config["server"], config["port"], keepalive=90)
        client.loop_forever()

    except KeyboardInterrupt:
        log("[EXIT] Shutting down gracefully...", level="LOG")
        # Send poison pill to stop MQTT publish worker
        mqtt_publish_queue.put((None, None))
        sys.exit(0)
    except Exception as e:
        log(f"[-] FATAL ERROR in main: {e}", level="LOG")
        log(f"[-] Traceback: {traceback.format_exc()}", level="LOG")
        sys.exit(1)

if __name__ == "__main__":
    main()
