#!/usr/bin/env python3
"""
IntelliStar Crash Monitor - Scans clientlog for errors and restarts XFree86
"""

import time
import paramiko
import re
from datetime import datetime
import asyncio
import discord
from discord.ext import tasks

# Configuration
SSH_HOST = "192.168.1.226"
SSH_USER = "root"
SSH_PASS = "root"
SSH_LOG_PATH = "/var/log/clientlog"  # Path to clientlog on device
CHECK_INTERVAL = 5  # seconds
RECOVERY_WAIT = 60  # seconds after killing process

# Discord Configuration
DISCORD_BOT_TOKEN = "MTQyOTE2ODUyNzIzNTA4ODQzNA.GdcC8o.eHrnrkE1tNi-B_QSKUsFUaUDOxs0gmD0Lh2MFs"
DISCORD_CHANNEL_ID = 1418786593191235684
DISCORD_USER_ID = 1403918926940868774

# Global Discord client
discord_client = None
HEADEND_NAME = "Unknown"  # Will be loaded from config.py
last_log_offset = 0  # Track where we left off in the log file


def load_headend_name_from_config(config_path="config.py"):
    """Extract headendName from config.py"""
    global HEADEND_NAME
    try:
        with open(config_path, "r", encoding="utf-8") as f:
            content = f.read()
        
        # Pattern to match: dsm.set('headendName','<name>', 0)
        pattern = r"dsm\.set\('headendName',\s*'([^']+)',\s*0\)"
        match = re.search(pattern, content)
        
        if match:
            HEADEND_NAME = match.group(1)
            print(f"Loaded headendName from config: {HEADEND_NAME}")
            return HEADEND_NAME
        else:
            print(f"Warning: Could not find headendName pattern in {config_path}")
            print("Using default: Unknown")
            return "Unknown"
    except FileNotFoundError:
        print(f"Warning: Config file not found: {config_path}")
        return "Unknown"
    except Exception as e:
        print(f"Error reading config: {e}")
        return "Unknown"


def get_clientlog_tail(client, lines=100):
    """SSH into device and get the tail of clientlog"""
    try:
        stdin, stdout, stderr = client.exec_command(f"tail -n {lines} {SSH_LOG_PATH}")
        exit_status = stdout.channel.recv_exit_status()
        
        if exit_status == 0:
            log_content = stdout.read().decode('utf-8', errors='ignore')
            return log_content
        else:
            error = stderr.read().decode()
            print(f"Error reading log: {error}")
            return None
    except Exception as e:
        print(f"Error getting clientlog: {e}")
        return None


def check_for_crash_patterns(log_content):
    """
    Scan log content for crash patterns
    Returns True if crash detected
    """
    if not log_content:
        return False
    
    # Patterns that indicate a crash
    crash_patterns = [
        r"\[ERR\].*runRenderScript",  # Corba interface error
        r"\[ERR\].*Traceback",  # Python traceback
        r"\[ERR\].*AttributeError",  # Missing attribute
        r"\[ERR\].*TypeError",  # Type error
        r"\[ERR\].*NameError",  # Undefined variable
        r"\[ERR\].*RuntimeError",  # Runtime exception
        r"\[ERR\].*IndexError",  # Index out of bounds
        r"Segmentation fault",  # Segfault
        r"Bus error",  # Bus error
        r"Core dumped",  # Core dump
    ]
    
    for pattern in crash_patterns:
        if re.search(pattern, log_content, re.IGNORECASE):
            print(f"Crash pattern detected: {pattern}")
            return True
    
    return False


async def send_discord_message():
    """Actually send the Discord message via bot"""
    try:
        channel = discord_client.get_channel(DISCORD_CHANNEL_ID)
        if not channel:
            print(f"Error: Could not find channel {DISCORD_CHANNEL_ID}")
            return
        
        # Get current timestamp
        now = datetime.now()
        timestamp = now.strftime("%m/%d/%Y at %H:%M")
        
        # Create embed with red color - use dynamic headend name
        embed = discord.Embed(
            title="Man Down!",
            description=f"Star WXS - {HEADEND_NAME}\nHas crashed! Fixing...",
            color=0xE74C3C  # Red color
        )
        embed.set_footer(text=f"Rainwater_i1_Downdetector - {timestamp}")
        
        # Send message with ping
        await channel.send(f"<@{DISCORD_USER_ID}>", embed=embed)
        print("Discord notification sent successfully")
        
    except Exception as e:
        print(f"Error sending Discord notification: {e}")


class MonitorBot(discord.Client):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.ssh_client = None
    
    async def on_ready(self):
        print(f'Discord bot logged in as {self.user}')
        print(f'Bot is in {len(self.guilds)} server(s)')
        
        # Establish SSH connection
        try:
            self.ssh_client = paramiko.SSHClient()
            self.ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            self.ssh_client.connect(SSH_HOST, username=SSH_USER, password=SSH_PASS, timeout=10)
            print(f"SSH connection established to {SSH_HOST}")
        except Exception as e:
            print(f"SSH connection error: {e}")
        
        # Start the monitoring task
        self.monitor_task.start()
    
    @tasks.loop(seconds=CHECK_INTERVAL)
    async def monitor_task(self):
        """Main monitoring loop - scan clientlog"""
        try:
            if not self.ssh_client:
                print("SSH not connected, skipping check...")
                return
            
            # Get recent logs
            log_content = get_clientlog_tail(self.ssh_client, lines=50)
            
            if log_content and check_for_crash_patterns(log_content):
                print("=" * 50)
                print("ALERT: Crash detected in clientlog!")
                print("=" * 50)
                
                # Send Discord notification
                await send_discord_message()
                
                # Kill XFree86 process
                if ssh_kill_xfree86(self.ssh_client):
                    print(f"Waiting {RECOVERY_WAIT} seconds for system recovery...")
                    await asyncio.sleep(RECOVERY_WAIT)
                    print("Resuming monitoring...")
                else:
                    print("Failed to kill process. Waiting 10 seconds before retry...")
                    await asyncio.sleep(10)
            else:
                # Normal operation - no crash found
                print(".", end="", flush=True)
                
        except Exception as e:
            print(f"\nError in monitoring loop: {e}")
            print("Waiting 5 seconds before retry...")
            await asyncio.sleep(5)
    
    @monitor_task.before_loop
    async def before_monitor_task(self):
        await self.wait_until_ready()


def ssh_kill_xfree86(client=None):
    """SSH into device and kill XFree86 process"""
    if client is None:
        client = paramiko.SSHClient()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        try:
            client.connect(SSH_HOST, username=SSH_USER, password=SSH_PASS, timeout=10)
        except Exception as e:
            print(f"Connection error: {e}")
            return False
    
    try:
        print(f"Connected to {SSH_HOST}")
        
        # Kill XFree86 process - FreeBSD 4.8 compatible
        print("Killing XFree86 process...")
        
        # First try killall (standard on FreeBSD)
        stdin, stdout, stderr = client.exec_command("killall XFree86")
        exit_status = stdout.channel.recv_exit_status()
        
        if exit_status == 0:
            print("Successfully killed XFree86 with killall")
        else:
            # Fallback: Find PID and kill directly (more compatible with older BSD)
            print("Trying alternate method...")
            stdin, stdout, stderr = client.exec_command("ps aux | grep XFree86 | grep -v grep | awk '{print $2}' | xargs kill -9")
            exit_status = stdout.channel.recv_exit_status()
            
            if exit_status == 0:
                print("Successfully killed XFree86 with kill -9")
            else:
                error = stderr.read().decode()
                print(f"Error killing XFree86: {error}")
        
        # Wait 10 seconds for X to restart, then run unclutter
        print("Waiting 10 seconds before running unclutter...")
        time.sleep(10)
        
        print("Running unclutter to hide cursor...")
        stdin, stdout, stderr = client.exec_command("DISPLAY=:0 unclutter -idle 0 -root &")
        # Don't wait for exit status since it's backgrounded
        
        print("Unclutter started successfully")
        
        return True
        
    except paramiko.AuthenticationException:
        print("Authentication failed")
        return False
    except paramiko.SSHException as e:
        print(f"SSH error: {e}")
        return False
    except Exception as e:
        print(f"Error: {e}")
        return False


def main():
    """Main entry point"""
    global discord_client
    
    print("IntelliStar Crash Monitor Starting...")
    print(f"Target: {SSH_HOST}")
    print(f"Log path: {SSH_LOG_PATH}")
    print(f"Check interval: {CHECK_INTERVAL}s")
    print(f"Recovery wait: {RECOVERY_WAIT}s")
    print("-" * 50)
    
    # Load headend name from config
    load_headend_name_from_config()
    
    print("Starting Discord bot and monitoring loop...")
    print()
    
    # Create and run Discord bot with minimal intents (we only send messages)
    intents = discord.Intents.default()
    discord_client = MonitorBot(intents=intents)
    
    try:
        discord_client.run(DISCORD_BOT_TOKEN)
    except KeyboardInterrupt:
        print("\n\nMonitoring stopped by user")
    except Exception as e:
        print(f"Error: {e}")


if __name__ == "__main__":
    print("Required packages: paramiko, discord.py")
    print("Install with: pip install paramiko discord.py")
    print()
    main()