import paramiko
import re
import time
import random
import signal
import subprocess
from threading import Thread, Lock

try:
    from pynput import keyboard
    KEYBOARD_AVAILABLE = True
except ImportError:
    KEYBOARD_AVAILABLE = False
    print("[BGM WARNING] pynput not installed - keyboard detection disabled")

# SSH connection details
HOST = '192.168.1.226'
USERNAME = 'root'
PASSWORD = 'root'
AUDIO_DIR = '/home/dgadmin/wxscan_music'
LOG_FILE = '/var/log/clientlog'

# Track list (WAV files)
TRACKS = [
    'ccm_track_01_122905_processed.wav',
    'ccm_track_02_122905_processed.wav',
    'ccm_track_03_122905_processed.wav',
    'ccm_track_04_122905_processed.wav',
    'ccm_track_05_122905_processed.wav',
    'ccm_track_06_122905_processed.wav',
    'jc2_track_03n.wav',
    'jc2_track_04n.wav',
    'jc_track_02n.wav',
    'ts_track_01n.wav',
    'ts_track_03n.wav',
    'ts_track_04n.wav',
    'ts_track_05n.wav',
    'ts_track_06n.wav',
    'ts_track_08n.wav',
    'tz_track_04n.wav',
    'tz_track_05n.wav',
    'tz_track_06n.wav',
    'tz_track_13n.wav',
    'tz_track_16n.wav',
    'tz_track_17n.wav',
]

last_played_track = None
shuffled_queue = []
is_playing = False
is_paused = False
force_play_triggered = False
current_playing_process = None
narration_lock = Lock()

def create_ssh_client():
    """Create and return an SSH client connection"""
    client = paramiko.SSHClient()
    client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
    client.connect(HOST, username=USERNAME, password=PASSWORD)
    return client

def get_next_shuffled_track():
    """Get next track from shuffled queue, never same as last played"""
    global shuffled_queue, last_played_track
    
    # Regenerate shuffled queue if empty
    if not shuffled_queue:
        shuffled_queue = TRACKS.copy()
        random.shuffle(shuffled_queue)
    
    # Get next track and ensure it's not the same as last played
    while shuffled_queue:
        track = shuffled_queue.pop(0)
        if track != last_played_track:
            last_played_track = track
            return track
        # If same as last, skip and try next
    
    # Fallback: regenerate and pick first
    shuffled_queue = TRACKS.copy()
    random.shuffle(shuffled_queue)
    track = shuffled_queue.pop(0)
    last_played_track = track
    return track

def play_audio(filename):
    """Play audio file on the remote system"""
    global is_playing, is_paused, current_playing_process
    try:
        client = create_ssh_client()
        # Use absolute path instead of cd
        command = f'play {AUDIO_DIR}/{filename}'
        print(f"[BGM] ▶ Playing: {filename}")
        print(f"[BGM] Command: {command}")
        stdin, stdout, stderr = client.exec_command(command)
        
        # Store process info for pause/resume capability
        current_playing_process = (client, stdin, stdout, stderr)
        
        # Wait for command to complete
        exit_code = stdout.channel.recv_exit_status()
        
        # Read stderr and stdout
        error = stderr.read().decode().strip()
        output = stdout.read().decode().strip()
        
        if exit_code != 0:
            print(f"[BGM ERROR] Exit code {exit_code}: {error}")
        elif error:
            print(f"[BGM ERROR] {error}")
        else:
            print(f"[BGM ✓] Successfully played {filename}")
        
        if output:
            print(f"[BGM OUTPUT] {output}")
        
        client.close()
    except Exception as e:
        print(f"[BGM ERROR] Failed to play {filename}: {e}")
    finally:
        is_playing = False
        is_paused = False
        current_playing_process = None

def on_key_press(key):
    """Handle keyboard input - press 'f' to force play music"""
    global force_play_triggered
    try:
        if hasattr(key, 'char') and key.char == 'f':
            print("[BGM] ✓ [F] KEY PRESSED - Force playing music!")
            force_play_triggered = True
    except AttributeError:
        pass

def pause_bgm():
    """Pause the currently playing BGM"""
    global is_paused, current_playing_process
    with narration_lock:
        if is_playing and not is_paused and current_playing_process:
            try:
                print("[BGM PAUSE] ⏸ Pausing background music for narration...")
                is_paused = True
                # Try to pause via SSH by killing with SIGSTOP
                try:
                    client, stdin, stdout, stderr = current_playing_process
                    # Send EOF to signal pause (platform dependent)
                    stdin.write(b'\x1a')
                    stdin.flush()
                except:
                    pass  # Not all platforms support this
            except Exception as e:
                print(f"[BGM ERROR] Failed to pause: {e}")

def resume_bgm():
    """Resume the paused BGM"""
    global is_paused
    with narration_lock:
        if is_paused:
            try:
                print("[BGM RESUME] ▶ Resuming background music...")
                is_paused = False
                # Note: Full pause/resume is complex with SSH, so we'll just set flag
                # The main loop will restart the track if needed
            except Exception as e:
                print(f"[BGM ERROR] Failed to resume: {e}")

def start_keyboard_listener():
    """Start listening for keyboard input in a background thread"""
    if not KEYBOARD_AVAILABLE:
        return
    
    try:
        listener = keyboard.Listener(on_press=on_key_press)
        listener.start()
        print("[BGM] Keyboard listener started - Press 'F' to force play music")
    except Exception as e:
        print(f"[BGM WARNING] Could not start keyboard listener: {e}")

def monitor_and_play():
    """Monitor clientlog for AttributeError and play background music on loop"""
    global is_playing, force_play_triggered, is_paused
    
    print("[BGM] Connecting to monitor log...")
    
    # Pattern to match the AttributeError - exact pattern from your log
    music_pattern = r"AttributeError: 'module' object has no attribute 'getBackgroundMusicList'"
    # Pattern to detect when narration (vocallocal) is about to play
    narration_pattern = r"\[PLAYING\]|\[warn\] File /audio/vocalLocal.*\.wav|playman\.py.*\[info\].*playing"
    
    try:
        client = create_ssh_client()
        
        # Execute tail -f command on clientlog
        stdin, stdout, stderr = client.exec_command(f'tail -f {LOG_FILE}')
        
        print(f"[BGM] ✓ Connected to {HOST}")
        print(f"[BGM] Monitoring {LOG_FILE} for AttributeError...")
        print("[BGM] Music will start when error is detected\n")
        print("[BGM] TIP: Press 'F' to force-start music at any time\n")
        
        # Start keyboard listener in background
        start_keyboard_listener()
        
        music_playing = False
        error_count = 0
        last_check = time.time()
        
        # Read output line by line
        for line in iter(stdout.readline, ''):
            line = line.strip()
            
            if line:
                print(f"[LOG] {line}")
            
            # PAUSE BGM: Detect when narration from vocallocal_cuer is about to play
            if re.search(narration_pattern, line) and is_playing and not is_paused:
                print("[BGM] Narration detected - PAUSING BGM...")
                pause_bgm()
                time.sleep(0.5)  # Give it a moment to pause
            
            # Check if user pressed 'F' key to force play
            if force_play_triggered and not is_playing:
                print("\n[BGM] ✓ FORCE PLAY triggered by keyboard")
                print(f"[BGM] is_playing={is_playing}, music_playing={music_playing}")
                track = get_next_shuffled_track()
                print(f"[BGM] Selected track: {track}")
                is_playing = True
                audio_thread = Thread(target=play_audio, args=(track,))
                audio_thread.daemon = True
                audio_thread.start()
                print("[BGM] ✓ Audio thread started")
                force_play_triggered = False
                music_playing = True
            
            # Check if this line contains the AttributeError
            if re.search(music_pattern, line):
                error_count += 1
                print(f"\n[BGM] ✓ ATTRIBUTEERROR DETECTED (#{error_count}) - Triggering music!")
                
                if not music_playing:
                    print("[BGM] Starting first track...")
                    music_playing = True
                
                # Play music in loop if not already playing
                if not is_playing:
                    track = get_next_shuffled_track()
                    print(f"[BGM] Selected track: {track}")
                    print(f"[BGM] Starting playback (is_playing was: {is_playing})")
                    is_playing = True
                    audio_thread = Thread(target=play_audio, args=(track,))
                    audio_thread.daemon = True
                    audio_thread.start()
                    print("[BGM] ✓ Audio thread started")
                else:
                    print(f"[BGM] Already playing, skipping (is_playing={is_playing})")
            
            # CONTINUOUS MUSIC LOOP: Keep playing tracks if music_playing is True
            # Check every 0.5 seconds if we should start the next track
            now = time.time()
            if now - last_check >= 0.5:
                last_check = now
                
                if music_playing and not is_playing and not is_paused:
                    # Music should be playing but nothing is playing - start next track
                    track = get_next_shuffled_track()
                    print(f"[BGM LOOP] Starting next track: {track}")
                    is_playing = True
                    audio_thread = Thread(target=play_audio, args=(track,))
                    audio_thread.daemon = True
                    audio_thread.start()
    
    except KeyboardInterrupt:
        print("\n[BGM] Stopping log monitor...")
    except Exception as e:
        print(f"[BGM ERROR] Monitoring failed: {e}")
        import traceback
        traceback.print_exc()
    finally:
        try:
            client.close()
            print("[BGM] Connection closed")
        except:
            pass

if __name__ == "__main__":
    print("=" * 60)
    print("Background Music (BGM) Worker")
    print("Log Monitor + Music Player")
    print("=" * 60)
    print(f"Host: {HOST}")
    print(f"Log File: {LOG_FILE}")
    print(f"Audio Directory: {AUDIO_DIR}")
    print("=" * 60 + "\n")
    
    monitor_and_play()
